"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.addColumn = addColumn;
exports.getStateColumnActions = getStateColumnActions;
exports.moveColumn = moveColumn;
exports.removeColumn = removeColumn;
var _common = require("../../../../common");
var _popularize_field = require("../../../utils/popularize_field");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

/**
 * Helper function to provide a fallback to a single _source column if the given array of columns
 * is empty, and removes _source if there are more than 1 columns given
 * @param columns
 * @param useNewFieldsApi should a new fields API be used
 */
function buildColumns(columns, useNewFieldsApi = false) {
  if (columns.length > 1 && columns.indexOf('_source') !== -1) {
    return columns.filter(col => col !== '_source');
  } else if (columns.length !== 0) {
    return columns;
  }
  return useNewFieldsApi ? [] : ['_source'];
}
function addColumn(columns, columnName, useNewFieldsApi) {
  if (columns.includes(columnName)) {
    return columns;
  }
  return buildColumns([...columns, columnName], useNewFieldsApi);
}
function removeColumn(columns, columnName, useNewFieldsApi) {
  if (!columns.includes(columnName)) {
    return columns;
  }
  return buildColumns(columns.filter(col => col !== columnName), useNewFieldsApi);
}
function moveColumn(columns, columnName, newIndex) {
  if (newIndex < 0 || newIndex >= columns.length || !columns.includes(columnName)) {
    return columns;
  }
  const modifiedColumns = [...columns];
  modifiedColumns.splice(modifiedColumns.indexOf(columnName), 1); // remove at old index
  modifiedColumns.splice(newIndex, 0, columnName); // insert before new index
  return modifiedColumns;
}
function getStateColumnActions({
  capabilities,
  config,
  dataView,
  dataViews,
  useNewFieldsApi,
  setAppState,
  columns,
  sort
}) {
  function onAddColumn(columnName) {
    (0, _popularize_field.popularizeField)(dataView, columnName, dataViews, capabilities);
    const nextColumns = addColumn(columns || [], columnName, useNewFieldsApi);
    const defaultOrder = config.get(_common.SORT_DEFAULT_ORDER_SETTING);
    const nextSort = columnName === '_score' && !(sort !== null && sort !== void 0 && sort.length) ? [['_score', defaultOrder]] : sort;
    setAppState({
      columns: nextColumns,
      sort: nextSort
    });
  }
  function onRemoveColumn(columnName) {
    (0, _popularize_field.popularizeField)(dataView, columnName, dataViews, capabilities);
    const nextColumns = removeColumn(columns || [], columnName, useNewFieldsApi);
    // The state's sort property is an array of [sortByColumn,sortDirection]
    const nextSort = sort && sort.length ? sort.filter(subArr => subArr[0] !== columnName) : [];
    setAppState({
      columns: nextColumns,
      sort: nextSort
    });
  }
  function onMoveColumn(columnName, newIndex) {
    const nextColumns = moveColumn(columns || [], columnName, newIndex);
    setAppState({
      columns: nextColumns
    });
  }
  function onSetColumns(nextColumns, hideTimeColumn) {
    // The next line should be gone when classic table will be removed
    const actualColumns = !hideTimeColumn && dataView.timeFieldName && dataView.timeFieldName === nextColumns[0] ? (nextColumns || []).slice(1) : nextColumns;
    setAppState({
      columns: actualColumns
    });
  }
  return {
    onAddColumn,
    onRemoveColumn,
    onMoveColumn,
    onSetColumns
  };
}