"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.RecordRawType = void 0;
exports.getDataStateContainer = getDataStateContainer;
var _rxjs = require("rxjs");
var _common = require("@kbn/inspector-plugin/common");
var _get_data_view_by_text_based_query_lang = require("../utils/get_data_view_by_text_based_query_lang");
var _is_text_based_query = require("../utils/is_text_based_query");
var _get_raw_record_type = require("../utils/get_raw_record_type");
var _common2 = require("../../../../common");
var _types = require("../../types");
var _validate_time_range = require("../utils/validate_time_range");
var _fetch_all = require("../utils/fetch_all");
var _use_saved_search_messages = require("../hooks/use_saved_search_messages");
var _get_fetch_observable = require("../utils/get_fetch_observable");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */
let RecordRawType;
exports.RecordRawType = RecordRawType;
(function (RecordRawType) {
  RecordRawType["DOCUMENT"] = "document";
  RecordRawType["PLAIN"] = "plain";
})(RecordRawType || (exports.RecordRawType = RecordRawType = {}));
/**
 * Container responsible for fetching of data in Discover Main
 * Either by triggering requests to Elasticsearch directly, or by
 * orchestrating unified plugins / components like the histogram
 */
function getDataStateContainer({
  services,
  searchSessionManager,
  getAppState,
  getSavedSearch,
  setDataView
}) {
  const {
    data,
    uiSettings,
    toastNotifications
  } = services;
  const {
    timefilter
  } = data.query.timefilter;
  const inspectorAdapters = {
    requests: new _common.RequestAdapter()
  };
  const appState = getAppState();
  const recordRawType = (0, _get_raw_record_type.getRawRecordType)(appState.query);
  /**
   * The observable to trigger data fetching in UI
   * By refetch$.next('reset') rows and fieldcounts are reset to allow e.g. editing of runtime fields
   * to be processed correctly
   */
  const refetch$ = new _rxjs.Subject();
  const getInitialFetchStatus = () => {
    const shouldSearchOnPageLoad = uiSettings.get(_common2.SEARCH_ON_PAGE_LOAD_SETTING) || getSavedSearch().id !== undefined || !timefilter.getRefreshInterval().pause || searchSessionManager.hasSearchSessionIdInURL();
    return shouldSearchOnPageLoad ? _types.FetchStatus.LOADING : _types.FetchStatus.UNINITIALIZED;
  };

  /**
   * The observables the UI (aka React component) subscribes to get notified about
   * the changes in the data fetching process (high level: fetching started, data was received)
   */
  const initialState = {
    fetchStatus: getInitialFetchStatus(),
    recordRawType
  };
  const dataSubjects = {
    main$: new _rxjs.BehaviorSubject(initialState),
    documents$: new _rxjs.BehaviorSubject(initialState),
    totalHits$: new _rxjs.BehaviorSubject(initialState),
    availableFields$: new _rxjs.BehaviorSubject(initialState)
  };
  let autoRefreshDone;
  /**
   * handler emitted by `timefilter.getAutoRefreshFetch$()`
   * to notify when data completed loading and to start a new autorefresh loop
   */
  const setAutoRefreshDone = fn => {
    autoRefreshDone = fn;
  };
  const fetch$ = (0, _get_fetch_observable.getFetch$)({
    setAutoRefreshDone,
    data,
    main$: dataSubjects.main$,
    refetch$,
    searchSource: getSavedSearch().searchSource,
    searchSessionManager
  }).pipe((0, _rxjs.filter)(() => (0, _validate_time_range.validateTimeRange)(timefilter.getTime(), toastNotifications)), (0, _rxjs.tap)(() => inspectorAdapters.requests.reset()), (0, _rxjs.map)(val => ({
    reset: val === 'reset',
    searchSessionId: searchSessionManager.getNextSearchSessionId()
  })), (0, _rxjs.share)());
  let abortController;
  function subscribe() {
    const subscription = fetch$.subscribe(async ({
      reset,
      searchSessionId
    }) => {
      var _abortController;
      (_abortController = abortController) === null || _abortController === void 0 ? void 0 : _abortController.abort();
      abortController = new AbortController();
      const prevAutoRefreshDone = autoRefreshDone;
      await (0, _fetch_all.fetchAll)(dataSubjects, reset, {
        abortController,
        initialFetchStatus: getInitialFetchStatus(),
        inspectorAdapters,
        searchSessionId,
        services,
        getAppState,
        savedSearch: getSavedSearch(),
        useNewFieldsApi: !uiSettings.get(_common2.SEARCH_FIELDS_FROM_SOURCE)
      });

      // If the autoRefreshCallback is still the same as when we started i.e. there was no newer call
      // replacing this current one, call it to make sure we tell that the auto refresh is done
      // and a new one can be scheduled.
      if (autoRefreshDone === prevAutoRefreshDone) {
        var _autoRefreshDone;
        // if this function was set and is executed, another refresh fetch can be triggered
        (_autoRefreshDone = autoRefreshDone) === null || _autoRefreshDone === void 0 ? void 0 : _autoRefreshDone();
        autoRefreshDone = undefined;
      }
    });
    return () => {
      var _abortController2;
      (_abortController2 = abortController) === null || _abortController2 === void 0 ? void 0 : _abortController2.abort();
      subscription.unsubscribe();
    };
  }
  const fetchQuery = async resetQuery => {
    const query = getAppState().query;
    const currentDataView = getSavedSearch().searchSource.getField('index');
    if ((0, _is_text_based_query.isTextBasedQuery)(query)) {
      const nextDataView = await (0, _get_data_view_by_text_based_query_lang.getDataViewByTextBasedQueryLang)(query, currentDataView, services);
      if (nextDataView !== currentDataView) {
        setDataView(nextDataView);
      }
    }
    if (resetQuery) {
      refetch$.next('reset');
    } else {
      refetch$.next(undefined);
    }
    return refetch$;
  };
  const reset = savedSearch => {
    const recordType = (0, _get_raw_record_type.getRawRecordType)(savedSearch.searchSource.getField('query'));
    (0, _use_saved_search_messages.sendResetMsg)(dataSubjects, getInitialFetchStatus(), recordType);
  };
  return {
    fetch: fetchQuery,
    fetch$,
    data$: dataSubjects,
    refetch$,
    subscribe,
    reset,
    inspectorAdapters,
    getInitialFetchStatus
  };
}