"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.fetchHitsInInterval = fetchHitsInInterval;
var _rxjs = require("rxjs");
var _public = require("@kbn/data-plugin/public");
var _build_data_record = require("../../../utils/build_data_record");
var _date_conversion = require("./date_conversion");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

/**
 * Fetch the hits between a given `interval` up to a maximum of `maxCount` documents.
 * The documents are sorted by `sort`
 *
 * The `searchSource` is assumed to have the appropriate data view
 * and filters set.
 */
async function fetchHitsInInterval(searchSource, timeField, sort, sortDir, interval, searchAfter, maxCount, nanosValue, anchorId) {
  var _rawResponse$hits;
  const range = {
    format: 'strict_date_optional_time'
  };
  const [start, stop] = interval;
  if (start) {
    range[sortDir === _public.SortDirection.asc ? 'gte' : 'lte'] = (0, _date_conversion.convertTimeValueToIso)(start, nanosValue);
  }
  if (stop) {
    range[sortDir === _public.SortDirection.asc ? 'lte' : 'gte'] = (0, _date_conversion.convertTimeValueToIso)(stop, nanosValue);
  }
  const fetch$ = searchSource.setField('size', maxCount).setField('query', {
    query: {
      bool: {
        must: {
          constant_score: {
            filter: {
              range: {
                [timeField]: range
              }
            }
          }
        },
        must_not: {
          ids: {
            values: [anchorId]
          }
        }
      }
    },
    language: 'lucene'
  }).setField('searchAfter', searchAfter).setField('sort', sort).setField('version', true).fetch$();
  const {
    rawResponse
  } = await (0, _rxjs.lastValueFrom)(fetch$);
  const dataView = searchSource.getField('index');
  const records = (_rawResponse$hits = rawResponse.hits) === null || _rawResponse$hits === void 0 ? void 0 : _rawResponse$hits.hits.map(hit => (0, _build_data_record.buildDataTableRecord)(hit, dataView));
  return records !== null && records !== void 0 ? records : [];
}