"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SurrDocType = void 0;
exports.fetchSurroundingDocs = fetchSurroundingDocs;
exports.updateSearchSource = updateSearchSource;
var _sorting = require("../utils/sorting");
var _date_conversion = require("../utils/date_conversion");
var _fetch_hits_in_interval = require("../utils/fetch_hits_in_interval");
var _generate_intervals = require("../utils/generate_intervals");
var _get_es_query_search_after = require("../utils/get_es_query_search_after");
var _get_es_query_sort = require("../utils/get_es_query_sort");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */
let SurrDocType;
exports.SurrDocType = SurrDocType;
(function (SurrDocType) {
  SurrDocType["SUCCESSORS"] = "successors";
  SurrDocType["PREDECESSORS"] = "predecessors";
})(SurrDocType || (exports.SurrDocType = SurrDocType = {}));
const DAY_MILLIS = 24 * 60 * 60 * 1000;

// look from 1 day up to 10000 days into the past and future
const LOOKUP_OFFSETS = [0, 1, 7, 30, 365, 10000].map(days => days * DAY_MILLIS);

/**
 * Fetch successor or predecessor documents of a given anchor document
 *
 * @param {SurrDocType} type - `successors` or `predecessors`
 * @param {DataView} dataView
 * @param {DataTableRecord} anchor - anchor record
 * @param {string} tieBreakerField - name of the tie breaker, the 2nd sort field
 * @param {SortDirection} sortDir - direction of sorting
 * @param {number} size - number of records to retrieve
 * @param {Filter[]} filters - to apply in the elastic query
 * @param {boolean} useNewFieldsApi
 * @returns {Promise<object[]>}
 */
async function fetchSurroundingDocs(type, dataView, anchor, tieBreakerField, sortDir, size, filters, data, useNewFieldsApi) {
  var _anchorRaw$fields, _anchorRaw$fields2, _anchorRaw$sort;
  if (typeof anchor !== 'object' || anchor === null || !size) {
    return [];
  }
  const timeField = dataView.timeFieldName;
  const searchSource = data.search.searchSource.createEmpty();
  updateSearchSource(searchSource, dataView, filters, Boolean(useNewFieldsApi));
  const sortDirToApply = type === SurrDocType.SUCCESSORS ? sortDir : (0, _sorting.reverseSortDir)(sortDir);
  const anchorRaw = anchor.raw;
  const nanos = dataView.isTimeNanosBased() ? (0, _date_conversion.extractNanos)((_anchorRaw$fields = anchorRaw.fields) === null || _anchorRaw$fields === void 0 ? void 0 : _anchorRaw$fields[timeField][0]) : '';
  const timeValueMillis = nanos !== '' ? (0, _date_conversion.convertIsoToMillis)((_anchorRaw$fields2 = anchorRaw.fields) === null || _anchorRaw$fields2 === void 0 ? void 0 : _anchorRaw$fields2[timeField][0]) : (_anchorRaw$sort = anchorRaw.sort) === null || _anchorRaw$sort === void 0 ? void 0 : _anchorRaw$sort[0];
  const intervals = (0, _generate_intervals.generateIntervals)(LOOKUP_OFFSETS, timeValueMillis, type, sortDir);
  let documents = [];
  for (const interval of intervals) {
    const remainingSize = size - documents.length;
    if (remainingSize <= 0) {
      break;
    }
    const searchAfter = (0, _get_es_query_search_after.getEsQuerySearchAfter)(type, documents, timeField, anchor, nanos, useNewFieldsApi);
    const sort = (0, _get_es_query_sort.getEsQuerySort)(timeField, tieBreakerField, sortDirToApply, nanos);
    const hits = await (0, _fetch_hits_in_interval.fetchHitsInInterval)(searchSource, timeField, sort, sortDirToApply, interval, searchAfter, remainingSize, nanos, anchor.raw._id);
    documents = type === SurrDocType.SUCCESSORS ? [...documents, ...hits] : [...hits.slice().reverse(), ...documents];
  }
  return documents;
}
function updateSearchSource(searchSource, dataView, filters, useNewFieldsApi) {
  if (useNewFieldsApi) {
    searchSource.removeField('fieldsFromSource');
    searchSource.setField('fields', [{
      field: '*',
      include_unmapped: 'true'
    }]);
  }
  return searchSource.setParent(undefined).setField('index', dataView).setField('filter', filters).setField('trackTotalHits', false);
}