"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SearchInterceptor = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _lodash = require("lodash");
var _rxjs = require("rxjs");
var _operators = require("rxjs/operators");
var _public = require("@kbn/bfetch-plugin/public");
var _public2 = require("@kbn/kibana-react-plugin/public");
var _public3 = require("@kbn/kibana-utils-plugin/public");
var _common = require("../../../common");
var _errors = require("../errors");
var _session = require("../session");
var _search_response_cache = require("./search_response_cache");
var _utils = require("./utils");
var _search_abort_controller = require("./search_abort_controller");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

const MAX_CACHE_ITEMS = 50;
const MAX_CACHE_SIZE_MB = 10;
class SearchInterceptor {
  /**
   * Observable that emits when the number of pending requests changes.
   * @internal
   */

  /**
   * @internal
   */

  /*
   * @internal
   */
  constructor(deps) {
    (0, _defineProperty2.default)(this, "uiSettingsSubs", []);
    (0, _defineProperty2.default)(this, "searchTimeout", void 0);
    (0, _defineProperty2.default)(this, "bFetchDisabled", void 0);
    (0, _defineProperty2.default)(this, "responseCache", new _search_response_cache.SearchResponseCache(MAX_CACHE_ITEMS, MAX_CACHE_SIZE_MB));
    (0, _defineProperty2.default)(this, "pendingCount$", new _rxjs.BehaviorSubject(0));
    (0, _defineProperty2.default)(this, "application", void 0);
    (0, _defineProperty2.default)(this, "docLinks", void 0);
    (0, _defineProperty2.default)(this, "batchedFetch", void 0);
    (0, _defineProperty2.default)(this, "showTimeoutErrorToast", (e, sessionId) => {
      this.deps.toasts.addDanger({
        title: 'Timed out',
        text: (0, _public2.toMountPoint)(e.getErrorMessage(this.application), {
          theme$: this.deps.theme.theme$
        })
      });
    });
    (0, _defineProperty2.default)(this, "showTimeoutErrorMemoized", (0, _lodash.memoize)(this.showTimeoutErrorToast, (_, sessionId) => {
      return sessionId;
    }));
    (0, _defineProperty2.default)(this, "showRestoreWarningToast", sessionId => {
      this.deps.toasts.addWarning({
        title: 'Your search session is still running',
        text: (0, _public2.toMountPoint)((0, _errors.SearchSessionIncompleteWarning)(this.docLinks), {
          theme$: this.deps.theme.theme$
        })
      }, {
        toastLifeTimeMs: 60000
      });
    });
    (0, _defineProperty2.default)(this, "showRestoreWarning", (0, _lodash.memoize)(this.showRestoreWarningToast));
    /**
     * Show one error notification per session.
     * @internal
     */
    (0, _defineProperty2.default)(this, "showTimeoutError", (e, sessionId) => {
      if (sessionId) {
        this.showTimeoutErrorMemoized(e, sessionId);
      } else {
        this.showTimeoutErrorToast(e, sessionId);
      }
    });
    this.deps = deps;
    this.deps.http.addLoadingCountSource(this.pendingCount$);
    this.deps.startServices.then(([coreStart]) => {
      this.application = coreStart.application;
      this.docLinks = coreStart.docLinks;
    });
    this.batchedFetch = deps.bfetch.batchedFunction({
      url: '/internal/bsearch'
    });
    this.searchTimeout = deps.uiSettings.get(_common.UI_SETTINGS.SEARCH_TIMEOUT);
    this.bFetchDisabled = deps.uiSettings.get(_public.DISABLE_BFETCH);
    this.uiSettingsSubs.push(deps.uiSettings.get$(_common.UI_SETTINGS.SEARCH_TIMEOUT).subscribe(timeout => {
      this.searchTimeout = timeout;
    }), deps.uiSettings.get$(_public.DISABLE_BFETCH).subscribe(bFetchDisabled => {
      this.bFetchDisabled = bFetchDisabled;
    }));
  }
  stop() {
    this.responseCache.clear();
    this.uiSettingsSubs.forEach(s => s.unsubscribe());
  }

  /*
   * @returns `TimeoutErrorMode` indicating what action should be taken in case of a request timeout based on license and permissions.
   * @internal
   */
  getTimeoutMode() {
    var _this$application$cap;
    return (_this$application$cap = this.application.capabilities.advancedSettings) !== null && _this$application$cap !== void 0 && _this$application$cap.save ? _errors.TimeoutErrorMode.CHANGE : _errors.TimeoutErrorMode.CONTACT;
  }
  createRequestHash$(request, options) {
    const {
      sessionId
    } = options;
    // Preference is used to ensure all queries go to the same set of shards and it doesn't need to be hashed
    // https://www.elastic.co/guide/en/elasticsearch/reference/current/search-shard-routing.html#shard-and-node-preference
    const {
      preference,
      ...params
    } = request.params || {};
    const hashOptions = {
      ...params,
      sessionId
    };
    if (!sessionId) return (0, _rxjs.of)(undefined); // don't use cache if doesn't belong to a session
    const sessionOptions = this.deps.session.getSearchOptions(options.sessionId);
    if (sessionOptions !== null && sessionOptions !== void 0 && sessionOptions.isRestore) return (0, _rxjs.of)(undefined); // don't use cache if restoring a session

    return (0, _rxjs.from)((0, _utils.createRequestHash)(hashOptions));
  }

  /*
   * @returns `Error` a search service specific error or the original error, if a specific error can't be recognized.
   * @internal
   */
  handleSearchError(e, options, isTimeout) {
    if (isTimeout || e.message === 'Request timed out') {
      // Handle a client or a server side timeout
      const err = new _errors.SearchTimeoutError(e, this.getTimeoutMode());

      // Show the timeout error here, so that it's shown regardless of how an application chooses to handle errors.
      // The timeout error is shown any time a request times out, or once per session, if the request is part of a session.
      this.showTimeoutError(err, options === null || options === void 0 ? void 0 : options.sessionId);
      return err;
    } else if (e instanceof _public3.AbortError || e instanceof _public.BfetchRequestError) {
      // In the case an application initiated abort, throw the existing AbortError, same with BfetchRequestErrors
      return e;
    } else if ((0, _errors.isEsError)(e)) {
      if ((0, _errors.isPainlessError)(e)) {
        return new _errors.PainlessError(e, options === null || options === void 0 ? void 0 : options.indexPattern);
      } else {
        return new _errors.EsError(e);
      }
    } else {
      return e instanceof Error ? e : new Error(e.message);
    }
  }
  getSerializableOptions(options) {
    const {
      sessionId,
      ...requestOptions
    } = options || {};
    const serializableOptions = {};
    const combined = {
      ...requestOptions,
      ...this.deps.session.getSearchOptions(sessionId)
    };
    if (combined.sessionId !== undefined) serializableOptions.sessionId = combined.sessionId;
    if (combined.isRestore !== undefined) serializableOptions.isRestore = combined.isRestore;
    if (combined.legacyHitsTotal !== undefined) serializableOptions.legacyHitsTotal = combined.legacyHitsTotal;
    if (combined.strategy !== undefined) serializableOptions.strategy = combined.strategy;
    if (combined.isStored !== undefined) serializableOptions.isStored = combined.isStored;
    if (combined.isSearchStored !== undefined) serializableOptions.isSearchStored = combined.isSearchStored;
    if (combined.executionContext !== undefined) {
      serializableOptions.executionContext = combined.executionContext;
    }
    return serializableOptions;
  }

  /**
   * @internal
   * Creates a new pollSearch that share replays its results
   */
  runSearch$({
    id,
    ...request
  }, options, searchAbortController) {
    const {
      sessionId,
      strategy
    } = options;
    const search = () => {
      var _searchTracker$before;
      const [{
        isSearchStored
      }, afterPoll] = (_searchTracker$before = searchTracker === null || searchTracker === void 0 ? void 0 : searchTracker.beforePoll()) !== null && _searchTracker$before !== void 0 ? _searchTracker$before : [{
        isSearchStored: false
      }, ({
        isSearchStored: boolean
      }) => {}];
      return this.runSearch({
        id,
        ...request
      }, {
        ...options,
        ...this.deps.session.getSearchOptions(sessionId),
        abortSignal: searchAbortController.getSignal(),
        isSearchStored
      }).then(result => {
        var _result$isStored;
        afterPoll({
          isSearchStored: (_result$isStored = result.isStored) !== null && _result$isStored !== void 0 ? _result$isStored : false
        });
        return result;
      }).catch(err => {
        afterPoll({
          isSearchStored: false
        });
        throw err;
      });
    };
    const searchTracker = this.deps.session.isCurrentSession(sessionId) ? this.deps.session.trackSearch({
      abort: () => searchAbortController.abort(),
      poll: async () => {
        if (id) {
          await search();
        }
      }
    }) : undefined;

    // track if this search's session will be send to background
    // if yes, then we don't need to cancel this search when it is aborted
    let isSavedToBackground = this.deps.session.isCurrentSession(sessionId) && this.deps.session.isStored();
    const savedToBackgroundSub = this.deps.session.isCurrentSession(sessionId) && this.deps.session.state$.pipe((0, _operators.skip)(1),
    // ignore any state, we are only interested in transition x -> BackgroundLoading
    (0, _operators.filter)(state => this.deps.session.isCurrentSession(sessionId) && state === _session.SearchSessionState.BackgroundLoading), (0, _operators.take)(1)).subscribe(() => {
      isSavedToBackground = true;
    });
    const sendCancelRequest = (0, _lodash.once)(() => {
      this.deps.http.delete(`/internal/search/${strategy}/${id}`, {
        version: '1'
      });
    });
    const cancel = () => id && !isSavedToBackground && sendCancelRequest();
    return (0, _common.pollSearch)(search, cancel, {
      pollInterval: this.deps.searchConfig.asyncSearch.pollInterval,
      ...options,
      abortSignal: searchAbortController.getSignal()
    }).pipe((0, _operators.tap)(response => {
      id = response.id;
      if ((0, _common.isCompleteResponse)(response)) {
        searchTracker === null || searchTracker === void 0 ? void 0 : searchTracker.complete();
      }
    }), (0, _operators.catchError)(e => {
      searchTracker === null || searchTracker === void 0 ? void 0 : searchTracker.error();
      cancel();
      return (0, _rxjs.throwError)(e);
    }), (0, _operators.finalize)(() => {
      searchAbortController.cleanup();
      if (savedToBackgroundSub) {
        savedToBackgroundSub.unsubscribe();
      }
    }),
    // This observable is cached in the responseCache.
    // Using shareReplay makes sure that future subscribers will get the final response

    (0, _operators.shareReplay)(1));
  }

  /**
   * @internal
   * @throws `AbortError` | `ErrorLike`
   */
  runSearch(request, options) {
    const {
      abortSignal
    } = options || {};
    if (this.bFetchDisabled) {
      const {
        executionContext,
        strategy,
        ...searchOptions
      } = this.getSerializableOptions(options);
      return this.deps.http.post(`/internal/search/${strategy}${request.id ? `/${request.id}` : ''}`, {
        version: '1',
        signal: abortSignal,
        context: executionContext,
        body: JSON.stringify({
          ...request,
          ...searchOptions
        })
      }).catch(e => {
        if (e !== null && e !== void 0 && e.body) {
          throw e.body;
        } else {
          throw e;
        }
      });
    } else {
      const {
        executionContext,
        ...rest
      } = options || {};
      return this.batchedFetch({
        request,
        options: this.getSerializableOptions({
          ...rest,
          executionContext: this.deps.executionContext.withGlobalContext(executionContext)
        })
      }, abortSignal);
    }
  }

  /**
   * @internal
   * Creates a new search observable and a corresponding search abort controller
   * If requestHash is defined, tries to return them first from cache.
   */
  getSearchResponse$(request, options, requestHash) {
    var _options$abortSignal;
    const cached = requestHash ? this.responseCache.get(requestHash) : undefined;
    const searchAbortController = (cached === null || cached === void 0 ? void 0 : cached.searchAbortController) || new _search_abort_controller.SearchAbortController(this.searchTimeout);

    // Create a new abort signal if one was not passed. This fake signal will never be aborted,
    // So the underlaying search will not be aborted, even if the other consumers abort.
    searchAbortController.addAbortSignal((_options$abortSignal = options.abortSignal) !== null && _options$abortSignal !== void 0 ? _options$abortSignal : new AbortController().signal);
    const response$ = (cached === null || cached === void 0 ? void 0 : cached.response$) || this.runSearch$(request, options, searchAbortController);
    if (requestHash && !this.responseCache.has(requestHash)) {
      this.responseCache.set(requestHash, {
        response$,
        searchAbortController
      });
    }
    return {
      response$,
      searchAbortController
    };
  }

  /**
   * Searches using the given `search` method. Overrides the `AbortSignal` with one that will abort
   * either when the request times out, or when the original `AbortSignal` is aborted. Updates
   * `pendingCount$` when the request is started/finalized.
   *
   * @param request
   * @options
   * @returns `Observable` emitting the search response or an error.
   */
  search({
    id,
    ...request
  }, options = {}) {
    const searchOptions = {
      ...options
    };
    if (!searchOptions.strategy) {
      searchOptions.strategy = _common.ENHANCED_ES_SEARCH_STRATEGY;
    }
    const {
      sessionId,
      abortSignal
    } = searchOptions;
    return this.createRequestHash$(request, searchOptions).pipe((0, _operators.switchMap)(requestHash => {
      const {
        searchAbortController,
        response$
      } = this.getSearchResponse$(request, searchOptions, requestHash);
      this.pendingCount$.next(this.pendingCount$.getValue() + 1);

      // Abort the replay if the abortSignal is aborted.
      // The underlaying search will not abort unless searchAbortController fires.
      const aborted$ = (abortSignal ? (0, _rxjs.fromEvent)(abortSignal, 'abort') : _rxjs.EMPTY).pipe((0, _operators.map)(() => {
        throw new _public3.AbortError();
      }));
      return response$.pipe((0, _operators.takeUntil)(aborted$), (0, _operators.catchError)(e => {
        return (0, _rxjs.throwError)(this.handleSearchError(e, searchOptions, searchAbortController.isTimeout()));
      }), (0, _operators.tap)(response => {
        const isSearchInScopeOfSession = sessionId && sessionId === this.deps.session.getSessionId();
        if (isSearchInScopeOfSession && this.deps.session.isRestore() && response.isRestored === false) {
          this.showRestoreWarning(sessionId);
        }
      }), (0, _operators.finalize)(() => {
        this.pendingCount$.next(this.pendingCount$.getValue() - 1);
      }));
    }));
  }
  showError(e) {
    if (e instanceof _public3.AbortError || e instanceof _errors.SearchTimeoutError) {
      // The SearchTimeoutError is shown by the interceptor in getSearchError (regardless of how the app chooses to handle errors)
      return;
    }
    const overrideDisplay = (0, _utils.getSearchErrorOverrideDisplay)({
      error: e,
      application: this.application
    });
    if (overrideDisplay) {
      this.deps.toasts.addDanger({
        title: overrideDisplay.title,
        text: (0, _public2.toMountPoint)(overrideDisplay.body, {
          theme$: this.deps.theme.theme$
        })
      });
    } else {
      this.deps.toasts.addError(e, {
        title: 'Search Error'
      });
    }
  }
}
exports.SearchInterceptor = SearchInterceptor;