"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.internalBulkResolve = internalBulkResolve;
exports.isBulkResolveError = isBulkResolveError;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _coreElasticsearchServerInternal = require("@kbn/core-elasticsearch-server-internal");
var _coreSavedObjectsServer = require("@kbn/core-saved-objects-server");
var _coreSavedObjectsBaseServerInternal = require("@kbn/core-saved-objects-base-server-internal");
var _coreUsageDataBaseServerInternal = require("@kbn/core-usage-data-base-server-internal");
var _pMap = _interopRequireDefault(require("p-map"));
var _utils = require("../utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

const MAX_CONCURRENT_RESOLVE = 10;

/**
 * Parameters for the internal bulkResolve function.
 *
 * @internal
 */

/** Type guard used in the repository. */
function isBulkResolveError(result) {
  return !!result.error;
}
async function internalBulkResolve(params) {
  const {
    registry,
    migrator,
    allowedTypes,
    client,
    serializer,
    getIndexForType,
    incrementCounterInternal,
    encryptionExtension,
    securityExtension,
    objects,
    options = {}
  } = params;
  if (objects.length === 0) {
    return {
      resolved_objects: []
    };
  }
  const allObjects = validateObjectTypes(objects, allowedTypes);
  const validObjects = allObjects.filter(_utils.isRight);
  const namespace = (0, _utils.normalizeNamespace)(options.namespace);
  const {
    migrationVersionCompatibility
  } = options;
  const aliasDocs = await fetchAndUpdateAliases(validObjects, client, serializer, getIndexForType, namespace);
  const docsToBulkGet = [];
  const aliasInfoArray = [];
  validObjects.forEach(({
    value: {
      type,
      id
    }
  }, i) => {
    const objectIndex = getIndexForType(type);
    docsToBulkGet.push({
      // attempt to find an exact match for the given ID
      _id: serializer.generateRawId(namespace, type, id),
      _index: objectIndex
    });
    const aliasDoc = aliasDocs[i];
    if (aliasDoc !== null && aliasDoc !== void 0 && aliasDoc.found) {
      const legacyUrlAlias = aliasDoc._source[_coreSavedObjectsBaseServerInternal.LEGACY_URL_ALIAS_TYPE];
      if (!legacyUrlAlias.disabled) {
        docsToBulkGet.push({
          // also attempt to find a match for the legacy URL alias target ID
          _id: serializer.generateRawId(namespace, type, legacyUrlAlias.targetId),
          _index: objectIndex
        });
        const {
          targetId,
          purpose
        } = legacyUrlAlias;
        aliasInfoArray.push({
          targetId,
          purpose
        });
        return;
      }
    }
    aliasInfoArray.push(undefined);
  });
  const bulkGetResponse = docsToBulkGet.length ? await client.mget({
    body: {
      docs: docsToBulkGet
    }
  }, {
    ignore: [404],
    meta: true
  }) : undefined;
  // exit early if a 404 isn't from elasticsearch
  if (bulkGetResponse && (0, _coreElasticsearchServerInternal.isNotFoundFromUnsupportedServer)({
    statusCode: bulkGetResponse.statusCode,
    headers: bulkGetResponse.headers
  })) {
    throw _coreSavedObjectsServer.SavedObjectsErrorHelpers.createGenericNotFoundEsUnavailableError();
  }
  let getResponseIndex = 0;
  let aliasInfoIndex = 0;

  // Helper function for the map block below
  async function getSavedObject(objectType, objectId, doc) {
    // Encryption
    // @ts-expect-error MultiGetHit._source is optional
    const object = (0, _utils.getSavedObjectFromSource)(registry, objectType, objectId, doc, {
      migrationVersionCompatibility
    });
    const migrated = migrator.migrateDocument(object, {
      allowDowngrade: true
    });
    if (!(encryptionExtension !== null && encryptionExtension !== void 0 && encryptionExtension.isEncryptableType(migrated.type))) {
      return migrated;
    }
    return encryptionExtension.decryptOrStripResponseAttributes(migrated);
  }

  // map function for pMap below
  const mapper = async either => {
    var _aliasMatchDoc;
    if ((0, _utils.isLeft)(either)) {
      return either.value;
    }
    const exactMatchDoc = bulkGetResponse === null || bulkGetResponse === void 0 ? void 0 : bulkGetResponse.body.docs[getResponseIndex++];
    let aliasMatchDoc;
    const aliasInfo = aliasInfoArray[aliasInfoIndex++];
    if (aliasInfo !== undefined) {
      aliasMatchDoc = bulkGetResponse === null || bulkGetResponse === void 0 ? void 0 : bulkGetResponse.body.docs[getResponseIndex++];
    }
    const foundExactMatch =
    // @ts-expect-error MultiGetHit._source is optional
    exactMatchDoc.found && (0, _utils.rawDocExistsInNamespace)(registry, exactMatchDoc, namespace);
    const foundAliasMatch =
    // @ts-expect-error MultiGetHit._source is optional
    ((_aliasMatchDoc = aliasMatchDoc) === null || _aliasMatchDoc === void 0 ? void 0 : _aliasMatchDoc.found) && (0, _utils.rawDocExistsInNamespace)(registry, aliasMatchDoc, namespace);
    const {
      type,
      id
    } = either.value;
    let result = null;
    try {
      if (foundExactMatch && foundAliasMatch) {
        result = {
          saved_object: await getSavedObject(type, id, exactMatchDoc),
          outcome: 'conflict',
          alias_target_id: aliasInfo.targetId,
          alias_purpose: aliasInfo.purpose
        };
        resolveCounter.recordOutcome(_coreUsageDataBaseServerInternal.REPOSITORY_RESOLVE_OUTCOME_STATS.CONFLICT);
      } else if (foundExactMatch) {
        result = {
          saved_object: await getSavedObject(type, id, exactMatchDoc),
          outcome: 'exactMatch'
        };
        resolveCounter.recordOutcome(_coreUsageDataBaseServerInternal.REPOSITORY_RESOLVE_OUTCOME_STATS.EXACT_MATCH);
      } else if (foundAliasMatch) {
        result = {
          saved_object: await getSavedObject(type, aliasInfo.targetId, aliasMatchDoc),
          outcome: 'aliasMatch',
          alias_target_id: aliasInfo.targetId,
          alias_purpose: aliasInfo.purpose
        };
        resolveCounter.recordOutcome(_coreUsageDataBaseServerInternal.REPOSITORY_RESOLVE_OUTCOME_STATS.ALIAS_MATCH);
      }
    } catch (error) {
      return {
        id,
        type,
        error: _coreSavedObjectsServer.SavedObjectsErrorHelpers.decorateGeneralError(error, 'Failed to migrate document to the latest version.')
      };
    }
    if (result !== null) {
      return result;
    }
    resolveCounter.recordOutcome(_coreUsageDataBaseServerInternal.REPOSITORY_RESOLVE_OUTCOME_STATS.NOT_FOUND);
    return {
      type,
      id,
      error: _coreSavedObjectsServer.SavedObjectsErrorHelpers.createGenericNotFoundError(type, id)
    };
  };
  const resolveCounter = new ResolveCounter();
  const resolvedObjects = await (0, _pMap.default)(allObjects, mapper, {
    concurrency: MAX_CONCURRENT_RESOLVE
  });
  incrementCounterInternal(_coreUsageDataBaseServerInternal.CORE_USAGE_STATS_TYPE, _coreUsageDataBaseServerInternal.CORE_USAGE_STATS_ID, resolveCounter.getCounterFields(), {
    refresh: false
  }).catch(() => {}); // if the call fails for some reason, intentionally swallow the error

  if (!securityExtension) return {
    resolved_objects: resolvedObjects
  };
  const redactedObjects = await (securityExtension === null || securityExtension === void 0 ? void 0 : securityExtension.authorizeAndRedactInternalBulkResolve({
    namespace,
    objects: resolvedObjects
  }));
  return {
    resolved_objects: redactedObjects
  };
}

/** Separates valid and invalid object types */
function validateObjectTypes(objects, allowedTypes) {
  return objects.map(object => {
    const {
      type,
      id
    } = object;
    if (!allowedTypes.includes(type)) {
      return (0, _utils.left)({
        type,
        id,
        error: _coreSavedObjectsServer.SavedObjectsErrorHelpers.createUnsupportedTypeError(type)
      });
    }
    return (0, _utils.right)(object);
  });
}
async function fetchAndUpdateAliases(validObjects, client, serializer, getIndexForType, namespace) {
  if (validObjects.length === 0) {
    return [];
  }
  const time = (0, _utils.getCurrentTime)();
  const bulkUpdateDocs = validObjects.map(({
    value: {
      type,
      id
    }
  }) => [{
    update: {
      _id: serializer.generateRawLegacyUrlAliasId(namespace, type, id),
      _index: getIndexForType(_coreSavedObjectsBaseServerInternal.LEGACY_URL_ALIAS_TYPE),
      _source: true
    }
  }, {
    script: {
      source: `
            if (ctx._source[params.type].disabled != true) {
              if (ctx._source[params.type].resolveCounter == null) {
                ctx._source[params.type].resolveCounter = 1;
              }
              else {
                ctx._source[params.type].resolveCounter += 1;
              }
              ctx._source[params.type].lastResolved = params.time;
              ctx._source.updated_at = params.time;
            }
          `,
      lang: 'painless',
      params: {
        type: _coreSavedObjectsBaseServerInternal.LEGACY_URL_ALIAS_TYPE,
        time
      }
    }
  }]).flat();
  const bulkUpdateResponse = await client.bulk({
    refresh: false,
    require_alias: true,
    body: bulkUpdateDocs
  });
  return bulkUpdateResponse.items.map(item => {
    var _item$update;
    // Map the bulk update response to the `_source` fields that were returned for each document
    return (_item$update = item.update) === null || _item$update === void 0 ? void 0 : _item$update.get;
  });
}
class ResolveCounter {
  constructor() {
    (0, _defineProperty2.default)(this, "record", new Map());
  }
  recordOutcome(outcome) {
    var _this$record$get;
    const val = (_this$record$get = this.record.get(outcome)) !== null && _this$record$get !== void 0 ? _this$record$get : 0;
    this.record.set(outcome, val + 1);
  }
  getCounterFields() {
    const counterFields = [];
    let total = 0;
    for (const [fieldName, incrementBy] of this.record.entries()) {
      total += incrementBy;
      counterFields.push({
        fieldName,
        incrementBy
      });
    }
    if (total > 0) {
      counterFields.push({
        fieldName: _coreUsageDataBaseServerInternal.REPOSITORY_RESOLVE_OUTCOME_STATS.TOTAL,
        incrementBy: total
      });
    }
    return counterFields;
  }
}