"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.incrementCounterInternal = void 0;
var _coreSavedObjectsServer = require("@kbn/core-saved-objects-server");
var _coreSavedObjectsBaseServerInternal = require("@kbn/core-saved-objects-base-server-internal");
var _constants = require("../../constants");
var _utils = require("../utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

const incrementCounterInternal = async ({
  type,
  id,
  counterFields,
  options
}, {
  registry,
  helpers,
  client,
  serializer,
  migrator
}) => {
  var _body$get$_source, _body$get, _body$get$_source$ref, _body$get2, _body$get3;
  const {
    common: commonHelper,
    preflight: preflightHelper,
    migration: migrationHelper
  } = helpers;
  const {
    migrationVersion,
    typeMigrationVersion,
    refresh = _constants.DEFAULT_REFRESH_SETTING,
    initialize = false,
    upsertAttributes,
    managed
  } = options;
  if (!id) {
    throw _coreSavedObjectsServer.SavedObjectsErrorHelpers.createBadRequestError('id cannot be empty'); // prevent potentially upserting a saved object with an empty ID
  }

  const normalizedCounterFields = counterFields.map(counterField => {
    /**
     * no counterField configs provided, instead a field name string was passed.
     * ie `incrementCounter(so_type, id, ['my_field_name'])`
     * Using the default of incrementing by 1
     */
    if (typeof counterField === 'string') {
      return {
        fieldName: counterField,
        incrementBy: initialize ? 0 : 1
      };
    }
    const {
      incrementBy = 1,
      fieldName
    } = counterField;
    return {
      fieldName,
      incrementBy: initialize ? 0 : incrementBy
    };
  });
  const namespace = (0, _utils.normalizeNamespace)(options.namespace);
  const time = (0, _utils.getCurrentTime)();
  let savedObjectNamespace;
  let savedObjectNamespaces;
  if (registry.isSingleNamespace(type) && namespace) {
    savedObjectNamespace = namespace;
  } else if (registry.isMultiNamespace(type)) {
    // note: this check throws an error if the object is found but does not exist in this namespace
    const preflightResult = await preflightHelper.preflightCheckNamespaces({
      type,
      id,
      namespace
    });
    if (preflightResult.checkResult === 'found_outside_namespace') {
      throw _coreSavedObjectsServer.SavedObjectsErrorHelpers.createConflictError(type, id);
    }
    if (preflightResult.checkResult === 'not_found') {
      // If an upsert would result in the creation of a new object, we need to check for alias conflicts too.
      // This takes an extra round trip to Elasticsearch, but this won't happen often.
      // TODO: improve performance by combining these into a single preflight check
      await preflightHelper.preflightCheckForUpsertAliasConflict(type, id, namespace);
    }
    savedObjectNamespaces = preflightResult.savedObjectNamespaces;
  }

  // attributes: { [counterFieldName]: incrementBy },
  const migrated = migrationHelper.migrateInputDocument({
    id,
    type,
    ...(savedObjectNamespace && {
      namespace: savedObjectNamespace
    }),
    ...(savedObjectNamespaces && {
      namespaces: savedObjectNamespaces
    }),
    attributes: {
      ...(upsertAttributes !== null && upsertAttributes !== void 0 ? upsertAttributes : {}),
      ...normalizedCounterFields.reduce((acc, counterField) => {
        const {
          fieldName,
          incrementBy
        } = counterField;
        acc[fieldName] = incrementBy;
        return acc;
      }, {})
    },
    migrationVersion,
    typeMigrationVersion,
    managed,
    updated_at: time
  });
  const raw = serializer.savedObjectToRaw(migrated);
  const body = await client.update({
    id: raw._id,
    index: commonHelper.getIndexForType(type),
    refresh,
    require_alias: true,
    _source: true,
    body: {
      script: {
        source: `
              for (int i = 0; i < params.counterFieldNames.length; i++) {
                def counterFieldName = params.counterFieldNames[i];
                def count = params.counts[i];

                if (ctx._source[params.type][counterFieldName] == null) {
                  ctx._source[params.type][counterFieldName] = count;
                }
                else {
                  ctx._source[params.type][counterFieldName] += count;
                }
              }
              ctx._source.updated_at = params.time;
            `,
        lang: 'painless',
        params: {
          counts: normalizedCounterFields.map(normalizedCounterField => normalizedCounterField.incrementBy),
          counterFieldNames: normalizedCounterFields.map(normalizedCounterField => normalizedCounterField.fieldName),
          time,
          type
        }
      },
      upsert: raw._source
    }
  });
  const {
    originId
  } = (_body$get$_source = (_body$get = body.get) === null || _body$get === void 0 ? void 0 : _body$get._source) !== null && _body$get$_source !== void 0 ? _body$get$_source : {};
  return {
    id,
    type,
    ...(savedObjectNamespaces && {
      namespaces: savedObjectNamespaces
    }),
    ...(originId && {
      originId
    }),
    updated_at: time,
    references: (_body$get$_source$ref = (_body$get2 = body.get) === null || _body$get2 === void 0 ? void 0 : _body$get2._source.references) !== null && _body$get$_source$ref !== void 0 ? _body$get$_source$ref : [],
    version: (0, _coreSavedObjectsBaseServerInternal.encodeHitVersion)(body),
    attributes: (_body$get3 = body.get) === null || _body$get3 === void 0 ? void 0 : _body$get3._source[type],
    ...(managed && {
      managed
    })
  };
};
exports.incrementCounterInternal = incrementCounterInternal;