"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.find = void 0;
var _lodash = require("lodash");
var _boom = _interopRequireDefault(require("@hapi/boom"));
var _constants = require("../../../common/constants");
var _api = require("../../../common/api");
var _error = require("../../common/error");
var _utils = require("../../common/utils");
var _utils2 = require("../utils");
var _authorization = require("../../authorization");
var _constants2 = require("../../common/constants");
var _runtime_types = require("../../../common/api/runtime_types");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Throws an error if the user tries to filter by more than MAX_CATEGORY_FILTER_LENGTH categories.
 */
function throwIfCategoryParamTooLong(category) {
  if (Array.isArray(category) && category.length > _constants.MAX_CATEGORY_FILTER_LENGTH) throw _boom.default.badRequest(`Too many categories provided. The maximum allowed is ${_constants.MAX_CATEGORY_FILTER_LENGTH}`);
}

/**
 * Retrieves a case and optionally its comments.
 *
 * @ignore
 */
const find = async (params, clientArgs) => {
  const {
    services: {
      caseService,
      licensingService
    },
    authorization,
    logger,
    savedObjectsSerializer,
    spaceId
  } = clientArgs;
  try {
    const queryParams = (0, _api.decodeWithExcessOrThrow)(_api.CasesFindRequestRt)(params);
    throwIfCategoryParamTooLong(queryParams.category);

    /**
     * Assign users to a case is only available to Platinum+
     */

    if (!(0, _lodash.isEmpty)(queryParams.assignees)) {
      const hasPlatinumLicenseOrGreater = await licensingService.isAtLeastPlatinum();
      if (!hasPlatinumLicenseOrGreater) {
        throw _boom.default.forbidden('In order to filter cases by assignees, you must be subscribed to an Elastic Platinum license');
      }
      licensingService.notifyUsage(_constants2.LICENSING_CASE_ASSIGNMENT_FEATURE);
    }
    const {
      filter: authorizationFilter,
      ensureSavedObjectsAreAuthorized
    } = await authorization.getAuthorizationFilter(_authorization.Operations.findCases);
    const queryArgs = {
      tags: queryParams.tags,
      reporters: queryParams.reporters,
      sortByField: queryParams.sortField,
      status: queryParams.status,
      severity: queryParams.severity,
      owner: queryParams.owner,
      from: queryParams.from,
      to: queryParams.to,
      assignees: queryParams.assignees,
      category: queryParams.category
    };
    const statusStatsOptions = (0, _utils2.constructQueryOptions)({
      ...queryArgs,
      status: undefined,
      authorizationFilter
    });
    const caseQueryOptions = (0, _utils2.constructQueryOptions)({
      ...queryArgs,
      authorizationFilter
    });
    const caseSearch = (0, _utils2.constructSearch)(queryParams.search, spaceId, savedObjectsSerializer);
    const [cases, statusStats] = await Promise.all([caseService.findCasesGroupedByID({
      caseOptions: {
        ...queryParams,
        ...caseQueryOptions,
        ...caseSearch,
        searchFields: (0, _utils.asArray)(queryParams.searchFields)
      }
    }), caseService.getCaseStatusStats({
      searchOptions: statusStatsOptions
    })]);
    ensureSavedObjectsAreAuthorized([...cases.casesMap.values()]);
    const res = (0, _utils.transformCases)({
      casesMap: cases.casesMap,
      page: cases.page,
      perPage: cases.perPage,
      total: cases.total,
      countOpenCases: statusStats.open,
      countInProgressCases: statusStats['in-progress'],
      countClosedCases: statusStats.closed
    });
    return (0, _runtime_types.decodeOrThrow)(_api.CasesFindResponseRt)(res);
  } catch (error) {
    throw (0, _error.createCaseError)({
      message: `Failed to find cases: ${JSON.stringify(params)}: ${error}`,
      error,
      logger
    });
  }
};
exports.find = find;