"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getTraceItems = getTraceItems;
var _common = require("@kbn/observability-plugin/common");
var _server = require("@kbn/observability-plugin/server");
var _apm = require("../../../common/es_fields/apm");
var _get_linked_children = require("../span_links/get_linked_children");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function getTraceItems(traceId, config, apmEventClient, start, end) {
  const maxTraceItems = config.ui.maxTraceItems;
  const excludedLogLevels = ['debug', 'info', 'warning'];
  const errorResponsePromise = apmEventClient.search('get_errors_docs', {
    apm: {
      events: [_common.ProcessorEvent.error]
    },
    body: {
      track_total_hits: false,
      size: maxTraceItems,
      _source: [_apm.TIMESTAMP, _apm.TRACE_ID, _apm.TRANSACTION_ID, _apm.PARENT_ID, _apm.SERVICE_NAME, _apm.ERROR_ID, _apm.ERROR_LOG_MESSAGE, _apm.ERROR_EXCEPTION, _apm.ERROR_GROUP_ID],
      query: {
        bool: {
          filter: [{
            term: {
              [_apm.TRACE_ID]: traceId
            }
          }, ...(0, _server.rangeQuery)(start, end)],
          must_not: {
            terms: {
              [_apm.ERROR_LOG_LEVEL]: excludedLogLevels
            }
          }
        }
      }
    }
  });
  const traceResponsePromise = apmEventClient.search('get_trace_docs', {
    apm: {
      events: [_common.ProcessorEvent.span, _common.ProcessorEvent.transaction]
    },
    body: {
      track_total_hits: Math.max(10000, maxTraceItems + 1),
      size: maxTraceItems,
      _source: [_apm.TIMESTAMP, _apm.TRACE_ID, _apm.PARENT_ID, _apm.SERVICE_NAME, _apm.SERVICE_ENVIRONMENT, _apm.AGENT_NAME, _apm.EVENT_OUTCOME, _apm.PROCESSOR_EVENT, _apm.TRANSACTION_DURATION, _apm.TRANSACTION_ID, _apm.TRANSACTION_NAME, _apm.TRANSACTION_TYPE, _apm.TRANSACTION_RESULT, _apm.FAAS_COLDSTART, _apm.SPAN_ID, _apm.SPAN_TYPE, _apm.SPAN_SUBTYPE, _apm.SPAN_ACTION, _apm.SPAN_NAME, _apm.SPAN_DURATION, _apm.SPAN_LINKS, _apm.SPAN_COMPOSITE_COUNT, _apm.SPAN_COMPOSITE_COMPRESSION_STRATEGY, _apm.SPAN_COMPOSITE_SUM, _apm.SPAN_SYNC, _apm.CHILD_ID],
      query: {
        bool: {
          filter: [{
            term: {
              [_apm.TRACE_ID]: traceId
            }
          }, ...(0, _server.rangeQuery)(start, end)],
          should: {
            exists: {
              field: _apm.PARENT_ID
            }
          }
        }
      },
      sort: [{
        _score: {
          order: 'asc'
        }
      }, {
        [_apm.TRANSACTION_DURATION]: {
          order: 'desc'
        }
      }, {
        [_apm.SPAN_DURATION]: {
          order: 'desc'
        }
      }]
    }
  });
  const [errorResponse, traceResponse, spanLinksCountById] = await Promise.all([errorResponsePromise, traceResponsePromise, (0, _get_linked_children.getSpanLinksCountById)({
    traceId,
    apmEventClient,
    start,
    end
  })]);
  const traceItemCount = traceResponse.hits.total.value;
  const exceedsMax = traceItemCount > maxTraceItems;
  const traceDocs = traceResponse.hits.hits.map(hit => hit._source);
  const errorDocs = errorResponse.hits.hits.map(hit => hit._source);
  return {
    exceedsMax,
    traceDocs,
    errorDocs,
    spanLinksCountById,
    traceItemCount,
    maxTraceItems
  };
}