"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.update = update;
var _moment = _interopRequireDefault(require("moment"));
var _boom = _interopRequireDefault(require("@hapi/boom"));
var _get_maintenance_window_from_raw = require("../get_maintenance_window_from_raw");
var _generate_maintenance_window_events = require("../generate_maintenance_window_events");
var _common = require("../../../common");
var _retry_if_conflicts = require("../../lib/retry_if_conflicts");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function update(context, params) {
  return await (0, _retry_if_conflicts.retryIfConflicts)(context.logger, `maintenanceWindowClient.update('${params.id})`, async () => {
    return await updateWithOCC(context, params);
  });
}
async function updateWithOCC(context, params) {
  const {
    savedObjectsClient,
    getModificationMetadata,
    logger
  } = context;
  const {
    id,
    title,
    enabled,
    duration,
    rRule
  } = params;
  try {
    const {
      attributes,
      id: fetchedId,
      version
    } = await savedObjectsClient.get(_common.MAINTENANCE_WINDOW_SAVED_OBJECT_TYPE, id);
    if (_moment.default.utc(attributes.expirationDate).isBefore(new Date())) {
      throw _boom.default.badRequest('Cannot edit archived maintenance windows');
    }
    const expirationDate = _moment.default.utc().add(1, 'year').toISOString();
    const modificationMetadata = await getModificationMetadata();
    let events = (0, _generate_maintenance_window_events.generateMaintenanceWindowEvents)({
      rRule: rRule || attributes.rRule,
      duration: typeof duration === 'number' ? duration : attributes.duration,
      expirationDate
    });
    if (!(0, _generate_maintenance_window_events.shouldRegenerateEvents)({
      maintenanceWindow: attributes,
      rRule,
      duration
    })) {
      events = (0, _generate_maintenance_window_events.mergeEvents)({
        oldEvents: attributes.events,
        newEvents: events
      });
    }
    const updatedAttributes = {
      ...attributes,
      ...(title ? {
        title
      } : {}),
      ...(rRule ? {
        rRule
      } : {}),
      ...(typeof duration === 'number' ? {
        duration
      } : {}),
      ...(typeof enabled === 'boolean' ? {
        enabled
      } : {}),
      expirationDate,
      events,
      updatedBy: modificationMetadata.updatedBy,
      updatedAt: modificationMetadata.updatedAt
    };

    // We are deleting and then creating rather than updating because SO.update
    // performs a partial update on the rRule, we would need to null out all of the fields
    // that are removed from a new rRule if that were the case.
    const result = await savedObjectsClient.create(_common.MAINTENANCE_WINDOW_SAVED_OBJECT_TYPE, updatedAttributes, {
      id: fetchedId,
      version,
      overwrite: true
    });
    return (0, _get_maintenance_window_from_raw.getMaintenanceWindowFromRaw)({
      attributes: result.attributes,
      id: result.id
    });
  } catch (e) {
    const errorMessage = `Failed to update maintenance window by id: ${id}, Error: ${e}`;
    logger.error(errorMessage);
    throw _boom.default.boomify(e, {
      message: errorMessage
    });
  }
}