"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.finish = finish;
var _moment = _interopRequireDefault(require("moment"));
var _boom = _interopRequireDefault(require("@hapi/boom"));
var _generate_maintenance_window_events = require("../generate_maintenance_window_events");
var _get_maintenance_window_date_and_status = require("../get_maintenance_window_date_and_status");
var _get_maintenance_window_from_raw = require("../get_maintenance_window_from_raw");
var _common = require("../../../common");
var _retry_if_conflicts = require("../../lib/retry_if_conflicts");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function finish(context, params) {
  return await (0, _retry_if_conflicts.retryIfConflicts)(context.logger, `maintenanceWindowClient.finish('${params.id})`, async () => {
    return await finishWithOCC(context, params);
  });
}
async function finishWithOCC(context, params) {
  const {
    savedObjectsClient,
    getModificationMetadata,
    logger
  } = context;
  const {
    id
  } = params;
  const modificationMetadata = await getModificationMetadata();
  const now = new Date();
  const expirationDate = _moment.default.utc(now).add(1, 'year').toDate();
  try {
    const {
      attributes,
      version,
      id: fetchedId
    } = await savedObjectsClient.get(_common.MAINTENANCE_WINDOW_SAVED_OBJECT_TYPE, id);

    // Generate new events with new expiration date
    const newEvents = (0, _generate_maintenance_window_events.generateMaintenanceWindowEvents)({
      rRule: attributes.rRule,
      duration: attributes.duration,
      expirationDate: expirationDate.toISOString()
    });

    // Merge it with the old events
    const events = (0, _generate_maintenance_window_events.mergeEvents)({
      newEvents,
      oldEvents: attributes.events
    });

    // Find the current event and status of the maintenance window
    const {
      status,
      index
    } = (0, _get_maintenance_window_date_and_status.getMaintenanceWindowDateAndStatus)({
      events,
      dateToCompare: now,
      expirationDate
    });

    // Throw if the maintenance window is not running, or event doesn't exist
    if (status !== _common.MaintenanceWindowStatus.Running) {
      throw _boom.default.badRequest('Cannot finish maintenance window that is not running');
    }
    if (typeof index !== 'number' || !events[index]) {
      throw _boom.default.badRequest('Cannot find maintenance window event to finish');
    }

    // Update the running event to finish now
    const eventToFinish = {
      gte: events[index].gte,
      lte: now.toISOString()
    };

    // Update the events with the new finished event
    const eventsWithFinishedEvent = [...events];
    eventsWithFinishedEvent[index] = eventToFinish;
    const result = await savedObjectsClient.update(_common.MAINTENANCE_WINDOW_SAVED_OBJECT_TYPE, fetchedId, {
      events: eventsWithFinishedEvent,
      expirationDate: expirationDate.toISOString(),
      updatedAt: modificationMetadata.updatedAt,
      updatedBy: modificationMetadata.updatedBy
    }, {
      version
    });
    return (0, _get_maintenance_window_from_raw.getMaintenanceWindowFromRaw)({
      attributes: {
        ...attributes,
        ...result.attributes
      },
      id: result.id
    });
  } catch (e) {
    const errorMessage = `Failed to finish maintenance window by id: ${id}, Error: ${e}`;
    logger.error(errorMessage);
    throw _boom.default.boomify(e, {
      message: errorMessage
    });
  }
}