"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.shouldRegenerateEvents = exports.mergeEvents = exports.generateMaintenanceWindowEvents = void 0;
var _lodash = _interopRequireDefault(require("lodash"));
var _momentTimezone = _interopRequireDefault(require("moment-timezone"));
var _rrule = require("rrule");
var _rrule2 = require("../lib/rrule");
var _snooze = require("../lib/snooze");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const generateMaintenanceWindowEvents = ({
  rRule,
  expirationDate,
  duration
}) => {
  const {
    dtstart,
    until,
    wkst,
    byweekday,
    tzid,
    ...rest
  } = rRule;
  const startDate = (0, _snooze.utcToLocalUtc)(new Date(dtstart), tzid);
  const endDate = (0, _snooze.utcToLocalUtc)(new Date(expirationDate), tzid);
  const rRuleOptions = {
    ...rest,
    dtstart: startDate,
    until: until ? (0, _snooze.utcToLocalUtc)(new Date(until), tzid) : null,
    wkst: wkst ? _rrule.Weekday.fromStr(wkst) : null,
    byweekday: byweekday ? (0, _rrule2.parseByWeekday)(byweekday) : null
  };
  try {
    const recurrenceRule = new _rrule.RRule(rRuleOptions);
    const occurrenceDates = recurrenceRule.between(startDate, endDate, true);
    return occurrenceDates.map(date => {
      const utcDate = (0, _snooze.localUtcToUtc)(date, tzid);
      return {
        gte: utcDate.toISOString(),
        lte: _momentTimezone.default.utc(utcDate).add(duration, 'ms').toISOString()
      };
    });
  } catch (e) {
    throw new Error(`Failed to process RRule ${rRule}. Error: ${e}`);
  }
};

/**
 * Checks to see if we should regenerate maintenance window events.
 * Don't regenerate old events if the underlying RRule/duration did not change.
 */
exports.generateMaintenanceWindowEvents = generateMaintenanceWindowEvents;
const shouldRegenerateEvents = ({
  maintenanceWindow,
  rRule,
  duration
}) => {
  // If the rRule fails a deep equality check (there is a change), we should regenerate events
  if (rRule && !_lodash.default.isEqual(rRule, maintenanceWindow.rRule)) {
    return true;
  }
  // If the duration changes, we should regenerate events
  if (typeof duration === 'number' && duration !== maintenanceWindow.duration) {
    return true;
  }
  return false;
};

/**
 * Updates and merges the old events with the new events to preserve old modified events,
 * Unless the maintenance window was archived, then the old events are trimmed.
 */
exports.shouldRegenerateEvents = shouldRegenerateEvents;
const mergeEvents = ({
  oldEvents,
  newEvents
}) => {
  // If new events have more entries (expiration date got pushed), we merge the old into the new
  if (newEvents.length > oldEvents.length) {
    return [...oldEvents, ...newEvents.slice(-(newEvents.length - oldEvents.length))];
  }
  // If new events have less entries (maintenance window got archived), we trim the old events
  if (oldEvents.length > newEvents.length) {
    return oldEvents.slice(0, newEvents.length);
  }
  return oldEvents;
};
exports.mergeEvents = mergeEvents;