"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useChangePointResults = useChangePointResults;
var _react = require("react");
var _i18n = require("@kbn/i18n");
var _mlDatePicker = require("@kbn/ml-date-picker");
var _mlIsDefined = require("@kbn/ml-is-defined");
var _use_aiops_app_context = require("../../hooks/use_aiops_app_context");
var _use_data_source = require("../../hooks/use_data_source");
var _use_cancellable_search = require("../../hooks/use_cancellable_search");
var _constants = require("./constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function getChangePointDetectionRequestBody({
  index,
  fn,
  metricField,
  splitField,
  timeInterval,
  timeField,
  afterKey
}, query) {
  const timeSeriesAgg = {
    over_time: {
      date_histogram: {
        field: timeField,
        fixed_interval: timeInterval
      },
      aggs: {
        function_value: {
          [fn]: {
            field: metricField
          }
        }
      }
    },
    change_point_request: {
      change_point: {
        buckets_path: 'over_time>function_value'
      }
    },
    // Bucket selecting and sorting are only applicable for partitions
    ...((0, _mlIsDefined.isDefined)(splitField) ? {
      select: {
        bucket_selector: {
          buckets_path: {
            p_value: 'change_point_request.p_value'
          },
          script: 'params.p_value < 1'
        }
      },
      sort: {
        bucket_sort: {
          sort: [{
            'change_point_request.p_value': {
              order: 'asc'
            }
          }]
        }
      }
    } : {})
  };
  const aggregations = splitField ? {
    groupings: {
      composite: {
        size: _constants.COMPOSITE_AGG_SIZE,
        ...(afterKey !== undefined ? {
          after: {
            splitFieldTerm: afterKey
          }
        } : {}),
        sources: [{
          splitFieldTerm: {
            terms: {
              field: splitField
            }
          }
        }]
      },
      aggregations: timeSeriesAgg
    }
  } : timeSeriesAgg;
  return {
    params: {
      index,
      size: 0,
      body: {
        query,
        aggregations
      }
    }
  };
}
function useChangePointResults(fieldConfig, requestParams, query, splitFieldCardinality) {
  const {
    notifications: {
      toasts
    }
  } = (0, _use_aiops_app_context.useAiopsAppContext)();
  const {
    dataView
  } = (0, _use_data_source.useDataSource)();
  const refresh = (0, _mlDatePicker.useRefresh)();
  const [results, setResults] = (0, _react.useState)([]);
  /**
   * null also means the fetching has been complete
   */
  const [progress, setProgress] = (0, _react.useState)(null);
  const isSingleMetric = !(0, _mlIsDefined.isDefined)(fieldConfig.splitField);
  const totalAggPages = (0, _react.useMemo)(() => {
    return Math.ceil(Math.min(splitFieldCardinality !== null && splitFieldCardinality !== void 0 ? splitFieldCardinality : 0, _constants.SPLIT_FIELD_CARDINALITY_LIMIT) / _constants.COMPOSITE_AGG_SIZE);
  }, [splitFieldCardinality]);
  const {
    runRequest,
    cancelRequest
  } = (0, _use_cancellable_search.useCancellableSearch)();
  const reset = (0, _react.useCallback)(() => {
    cancelRequest();
    setResults([]);
  }, [cancelRequest]);
  const fetchResults = (0, _react.useCallback)(async (pageNumber = 1, afterKey) => {
    try {
      var _result$rawResponse$a, _result$rawResponse$a2, _result$rawResponse$a3, _result$rawResponse$a4, _result$rawResponse$a5, _result$rawResponse$a6;
      if (!isSingleMetric && !totalAggPages) {
        setProgress(null);
        return;
      }
      const requestPayload = getChangePointDetectionRequestBody({
        index: dataView.getIndexPattern(),
        fn: fieldConfig.fn,
        timeInterval: requestParams.interval,
        metricField: fieldConfig.metricField,
        timeField: dataView.timeFieldName,
        splitField: fieldConfig.splitField,
        afterKey
      }, query);
      const result = await runRequest(requestPayload);
      if (result === null) {
        setProgress(null);
        return;
      }
      const isFetchCompleted = !((_result$rawResponse$a = result.rawResponse.aggregations) !== null && _result$rawResponse$a !== void 0 && (_result$rawResponse$a2 = _result$rawResponse$a.groupings) !== null && _result$rawResponse$a2 !== void 0 && (_result$rawResponse$a3 = _result$rawResponse$a2.after_key) !== null && _result$rawResponse$a3 !== void 0 && _result$rawResponse$a3.splitFieldTerm && pageNumber < totalAggPages);
      const buckets = isSingleMetric ? [result.rawResponse.aggregations] : result.rawResponse.aggregations.groupings.buckets;
      setProgress(isFetchCompleted ? null : Math.min(Math.round(pageNumber / totalAggPages * 100), 100));
      let groups = buckets.map(v => {
        var _v$change_point_reque, _v$key;
        const changePointType = Object.keys(v.change_point_request.type)[0];
        const timeAsString = (_v$change_point_reque = v.change_point_request.bucket) === null || _v$change_point_reque === void 0 ? void 0 : _v$change_point_reque.key;
        const rawPValue = v.change_point_request.type[changePointType].p_value;
        return {
          ...(isSingleMetric ? {} : {
            group: {
              name: fieldConfig.splitField,
              value: v.key.splitFieldTerm
            }
          }),
          type: changePointType,
          p_value: rawPValue,
          timestamp: timeAsString,
          label: changePointType,
          reason: v.change_point_request.type[changePointType].reason,
          id: isSingleMetric ? 'single_metric' : `${fieldConfig.splitField}_${(_v$key = v.key) === null || _v$key === void 0 ? void 0 : _v$key.splitFieldTerm}`
        };
      }).filter(v => !_constants.EXCLUDED_CHANGE_POINT_TYPES.has(v.type));
      if (Array.isArray(requestParams.changePointType)) {
        groups = groups.filter(v => requestParams.changePointType.includes(v.type));
      }
      setResults(prev => {
        return (prev !== null && prev !== void 0 ? prev : []).concat(groups);
      });
      if (!isFetchCompleted && (_result$rawResponse$a4 = result.rawResponse.aggregations) !== null && _result$rawResponse$a4 !== void 0 && (_result$rawResponse$a5 = _result$rawResponse$a4.groupings) !== null && _result$rawResponse$a5 !== void 0 && (_result$rawResponse$a6 = _result$rawResponse$a5.after_key) !== null && _result$rawResponse$a6 !== void 0 && _result$rawResponse$a6.splitFieldTerm) {
        await fetchResults(pageNumber + 1, result.rawResponse.aggregations.groupings.after_key.splitFieldTerm);
      }
    } catch (e) {
      toasts.addError(e, {
        title: _i18n.i18n.translate('xpack.aiops.changePointDetection.fetchErrorTitle', {
          defaultMessage: 'Failed to fetch change points'
        })
      });
    }
  }, [runRequest, requestParams.interval, requestParams.changePointType, fieldConfig.fn, fieldConfig.metricField, fieldConfig.splitField, query, dataView, totalAggPages, toasts, isSingleMetric]);
  (0, _react.useEffect)(function fetchResultsOnInputChange() {
    setProgress(0);
    reset();
    if (fieldConfig.splitField && splitFieldCardinality === null) {
      // wait for cardinality to be resolved
      return;
    }
    fetchResults();
    return () => {
      cancelRequest();
    };
  }, [requestParams.interval, requestParams.changePointType, fieldConfig.fn, fieldConfig.metricField, fieldConfig.splitField, query, splitFieldCardinality, fetchResults, reset, cancelRequest, refresh]);
  return {
    results,
    isLoading: progress !== null,
    reset,
    progress
  };
}

/**
 * Response type for aggregation with composite agg pagination.
 * TODO: update type for the single metric
 */