"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.WatchVisualization = void 0;
var _react = _interopRequireWildcard(require("react"));
var _charts = require("@elastic/charts");
var _datemath = _interopRequireDefault(require("@kbn/datemath"));
var _momentTimezone = _interopRequireDefault(require("moment-timezone"));
var _eui = require("@elastic/eui");
var _i18nReact = require("@kbn/i18n-react");
var _visualize_options = require("../../../../models/visualize_options");
var _threshold_watch = require("../../../../models/watch/threshold_watch");
var _api = require("../../../../lib/api");
var _watch_context = require("../../watch_context");
var _agg_types = require("../../../../models/watch/agg_types");
var _comparators = require("../../../../models/watch/comparators");
var _components = require("../../../../components");
var _app_context = require("../../../../app_context");
function _getRequireWildcardCache(nodeInterop) { if (typeof WeakMap !== "function") return null; var cacheBabelInterop = new WeakMap(); var cacheNodeInterop = new WeakMap(); return (_getRequireWildcardCache = function (nodeInterop) { return nodeInterop ? cacheNodeInterop : cacheBabelInterop; })(nodeInterop); }
function _interopRequireWildcard(obj, nodeInterop) { if (!nodeInterop && obj && obj.__esModule) { return obj; } if (obj === null || typeof obj !== "object" && typeof obj !== "function") { return { default: obj }; } var cache = _getRequireWildcardCache(nodeInterop); if (cache && cache.has(obj)) { return cache.get(obj); } var newObj = {}; var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var key in obj) { if (key !== "default" && Object.prototype.hasOwnProperty.call(obj, key)) { var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null; if (desc && (desc.get || desc.set)) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } newObj.default = obj; if (cache) { cache.set(obj, newObj); } return newObj; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const customTheme = () => {
  return {
    lineSeriesStyle: {
      line: {
        strokeWidth: 3
      },
      point: {
        visible: false
      }
    }
  };
};
const getTimezone = config => {
  const DATE_FORMAT_CONFIG_KEY = 'dateFormat:tz';
  const isCustomTimezone = !config.isDefault(DATE_FORMAT_CONFIG_KEY);
  if (isCustomTimezone) {
    return config.get(DATE_FORMAT_CONFIG_KEY);
  }
  const detectedTimezone = _momentTimezone.default.tz.guess();
  if (detectedTimezone) {
    return detectedTimezone;
  }
  // default to UTC if we can't figure out the timezone
  return (0, _momentTimezone.default)().format('Z');
};
const getDomain = watch => {
  const VISUALIZE_TIME_WINDOW_MULTIPLIER = 5;
  const fromExpression = `now-${watch.timeWindowSize * VISUALIZE_TIME_WINDOW_MULTIPLIER}${watch.timeWindowUnit}`;
  const toExpression = 'now';
  const fromMoment = _datemath.default.parse(fromExpression);
  const toMoment = _datemath.default.parse(toExpression);
  const visualizeTimeWindowFrom = fromMoment ? fromMoment.valueOf() : 0;
  const visualizeTimeWindowTo = toMoment ? toMoment.valueOf() : 0;
  return {
    min: visualizeTimeWindowFrom,
    max: visualizeTimeWindowTo
  };
};
const getThreshold = watch => {
  return watch.threshold.slice(0, _comparators.comparators[watch.thresholdComparator].requiredValues);
};
const getTimeBuckets = (watch, timeBuckets) => {
  const domain = getDomain(watch);
  timeBuckets.setBounds(domain);
  return timeBuckets;
};
const WatchVisualization = () => {
  const {
    createTimeBuckets,
    theme,
    uiSettings
  } = (0, _app_context.useAppContext)();
  const {
    watch
  } = (0, _react.useContext)(_watch_context.WatchContext);
  const chartsTheme = theme.useChartsTheme();
  const {
    index,
    timeField,
    triggerIntervalSize,
    triggerIntervalUnit,
    aggType,
    aggField,
    termSize,
    termField,
    thresholdComparator,
    timeWindowSize,
    timeWindowUnit,
    groupBy,
    threshold
  } = watch;

  // Only recalculate the domain if the watch configuration changes. This prevents the visualization
  // request's resolution from re-triggering itself in an infinite loop.
  const domain = (0, _react.useMemo)(() => getDomain(watch), [watch]);
  const timeBuckets = createTimeBuckets();
  timeBuckets.setBounds(domain);
  const interval = timeBuckets.getInterval().expression;
  const visualizeOptions = new _visualize_options.VisualizeOptions({
    rangeFrom: domain.min,
    rangeTo: domain.max,
    interval,
    timezone: getTimezone(uiSettings)
  });

  // Fetching visualization data is independent of watch actions
  const watchWithoutActions = new _threshold_watch.ThresholdWatch({
    ...watch,
    actions: []
  });
  const {
    isInitialRequest,
    isLoading,
    data: watchVisualizationData,
    error,
    resendRequest: reload
  } = (0, _api.useGetWatchVisualizationData)(watchWithoutActions, visualizeOptions);
  (0, _react.useEffect)(() => {
    // Prevent sending a second request on initial render.
    if (isInitialRequest) {
      return;
    }
    reload();
  },
  // eslint-disable-next-line react-hooks/exhaustive-deps
  [index, timeField, triggerIntervalSize, triggerIntervalUnit, aggType, aggField, termSize, termField, thresholdComparator, timeWindowSize, timeWindowUnit, groupBy, threshold]);
  if (isInitialRequest && isLoading) {
    return /*#__PURE__*/_react.default.createElement(_eui.EuiEmptyPrompt, {
      title: /*#__PURE__*/_react.default.createElement(_eui.EuiLoadingChart, {
        size: "xl"
      }),
      body: /*#__PURE__*/_react.default.createElement(_eui.EuiText, {
        color: "subdued"
      }, /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
        id: "xpack.watcher.sections.watchEdit.loadingWatchVisualizationDescription",
        defaultMessage: "Loading watch visualization\u2026"
      }))
    });
  }
  if (error) {
    return /*#__PURE__*/_react.default.createElement(_react.Fragment, null, /*#__PURE__*/_react.default.createElement(_eui.EuiSpacer, {
      size: "l"
    }), /*#__PURE__*/_react.default.createElement(_components.SectionError, {
      title: /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
        id: "xpack.watcher.sections.watchEdit.errorLoadingWatchVisualizationTitle",
        defaultMessage: "Cannot load watch visualization"
      }),
      error: error
    }), /*#__PURE__*/_react.default.createElement(_eui.EuiSpacer, {
      size: "l"
    }));
  }
  if (watchVisualizationData) {
    const watchVisualizationDataKeys = Object.keys(watchVisualizationData);
    const timezone = getTimezone(uiSettings);
    const actualThreshold = getThreshold(watch);
    let maxY = actualThreshold[actualThreshold.length - 1];
    Object.values(watchVisualizationData).forEach(watchData => {
      watchData.forEach(([, y]) => {
        if (y > maxY) {
          maxY = y;
        }
      });
    });
    const dateFormatter = d => {
      return (0, _momentTimezone.default)(d).tz(timezone).format(getTimeBuckets(watch, createTimeBuckets()).getScaledDateFormat());
    };
    const aggLabel = _agg_types.aggTypes[watch.aggType].text;
    return /*#__PURE__*/_react.default.createElement("div", {
      "data-test-subj": "watchVisualizationChart"
    }, /*#__PURE__*/_react.default.createElement(_eui.EuiSpacer, {
      size: "l"
    }), watchVisualizationDataKeys.length ? /*#__PURE__*/_react.default.createElement(_charts.Chart, {
      size: ['100%', 300],
      renderer: "canvas"
    }, /*#__PURE__*/_react.default.createElement(_charts.Settings, {
      theme: [customTheme(), chartsTheme],
      xDomain: domain,
      showLegend: !!watch.termField,
      showLegendExtra: true,
      legendPosition: _charts.Position.Bottom
    }), /*#__PURE__*/_react.default.createElement(_charts.Axis, {
      id: "bottom",
      position: _charts.Position.Bottom,
      showOverlappingTicks: true,
      tickFormat: dateFormatter
    }), /*#__PURE__*/_react.default.createElement(_charts.Axis, {
      domain: {
        max: maxY,
        min: NaN
      },
      id: "left",
      title: aggLabel,
      position: _charts.Position.Left
    }), watchVisualizationDataKeys.map(key => {
      return /*#__PURE__*/_react.default.createElement(_charts.LineSeries, {
        key: key,
        id: key,
        xScaleType: _charts.ScaleType.Time,
        yScaleType: _charts.ScaleType.Linear,
        data: watchVisualizationData[key],
        xAccessor: 0,
        yAccessors: [1],
        timeZone: timezone
      });
    }), actualThreshold.map((_value, i) => {
      const specId = i === 0 ? 'threshold' : `threshold${i}`;
      return /*#__PURE__*/_react.default.createElement(_charts.LineAnnotation, {
        key: specId,
        id: specId,
        domainType: _charts.AnnotationDomainType.YDomain,
        dataValues: [{
          dataValue: watch.threshold[i],
          details: specId
        }]
      });
    })) : /*#__PURE__*/_react.default.createElement(_eui.EuiCallOut, {
      title: /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
        id: "xpack.watcher.thresholdPreviewChart.noDataTitle",
        defaultMessage: "No data"
      }),
      color: "warning"
    }, /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
      id: "xpack.watcher.thresholdPreviewChart.dataDoesNotExistTextMessage",
      defaultMessage: "Your index and condition did not return any data."
    })), /*#__PURE__*/_react.default.createElement(_eui.EuiSpacer, {
      size: "l"
    }));
  }
  return null;
};
exports.WatchVisualization = WatchVisualization;