"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useVisualizeAppState = void 0;
var _react = _interopRequireWildcard(require("react"));
var _lodash = require("lodash");
var _operators = require("rxjs/operators");
var _i18n = require("@kbn/i18n");
var _esQuery = require("@kbn/es-query");
var _public = require("@kbn/kibana-react-plugin/public");
var _public2 = require("@kbn/data-plugin/public");
var _migrate_legacy_query = require("../migrate_legacy_query");
var _utils = require("../utils");
var _create_visualize_app_state = require("../create_visualize_app_state");
var _constants = require("../../../../common/constants");
function _getRequireWildcardCache(nodeInterop) { if (typeof WeakMap !== "function") return null; var cacheBabelInterop = new WeakMap(); var cacheNodeInterop = new WeakMap(); return (_getRequireWildcardCache = function (nodeInterop) { return nodeInterop ? cacheNodeInterop : cacheBabelInterop; })(nodeInterop); }
function _interopRequireWildcard(obj, nodeInterop) { if (!nodeInterop && obj && obj.__esModule) { return obj; } if (obj === null || typeof obj !== "object" && typeof obj !== "function") { return { default: obj }; } var cache = _getRequireWildcardCache(nodeInterop); if (cache && cache.has(obj)) { return cache.get(obj); } var newObj = {}; var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var key in obj) { if (key !== "default" && Object.prototype.hasOwnProperty.call(obj, key)) { var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null; if (desc && (desc.get || desc.set)) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } newObj.default = obj; if (cache) { cache.set(obj, newObj); } return newObj; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

/**
 * This effect is responsible for instantiating the visualize app state container,
 * which is in sync with "_a" url param
 */
const useVisualizeAppState = (services, eventEmitter, instance) => {
  const [hasUnappliedChanges, setHasUnappliedChanges] = (0, _react.useState)(false);
  const [appState, setAppState] = (0, _react.useState)(null);
  (0, _react.useEffect)(() => {
    if (instance) {
      const stateDefaults = (0, _utils.visStateToEditorState)(instance, services);
      const byValue = !('savedVis' in instance);
      const {
        stateContainer,
        stopStateSync
      } = (0, _create_visualize_app_state.createVisualizeAppState)({
        stateDefaults,
        kbnUrlStateStorage: services.kbnUrlStateStorage,
        byValue
      });
      const currentAppState = stateContainer.getState();
      const onDirtyStateChange = ({
        isDirty
      }) => {
        if (!isDirty) {
          // it is important to update vis state with fresh data
          stateContainer.transitions.updateVisState((0, _utils.visStateToEditorState)(instance, services).vis);
        }
        setHasUnappliedChanges(isDirty);
      };
      eventEmitter.on('dirtyStateChange', onDirtyStateChange);
      const {
        filterManager,
        queryString
      } = services.data.query;
      // sync initial app state from state to managers
      filterManager.setAppFilters((0, _lodash.cloneDeep)(currentAppState.filters));
      queryString.setQuery((0, _migrate_legacy_query.migrateLegacyQuery)(currentAppState.query));

      // setup syncing of app filters between appState and query services
      const stopSyncingAppFilters = (0, _public2.connectToQueryState)(services.data.query, {
        set: ({
          filters,
          query
        }) => {
          stateContainer.transitions.set('filters', filters);
          stateContainer.transitions.set('query', query);
        },
        get: () => {
          return {
            filters: stateContainer.getState().filters,
            query: stateContainer.getState().query
          };
        },
        state$: stateContainer.state$.pipe((0, _operators.map)(state => ({
          filters: state.filters,
          query: state.query
        })))
      }, {
        filters: _esQuery.FilterStateStore.APP_STATE,
        query: true
      });

      // The savedVis is pulled from elasticsearch, but the appState is pulled from the url, with the
      // defaults applied. If the url was from a previous session which included modifications to the
      // appState then they won't be equal.
      if (!(0, _lodash.isEqual)(currentAppState.vis, stateDefaults.vis) || !(0, _lodash.isEqual)(currentAppState.query, stateDefaults.query) || !(0, _lodash.isEqual)(currentAppState.filters, stateDefaults.filters)) {
        var _instance$vis$data$se;
        const {
          aggs,
          ...visState
        } = currentAppState.vis;
        const query = currentAppState.query;
        const filter = currentAppState.filters;
        const visSearchSource = ((_instance$vis$data$se = instance.vis.data.searchSource) === null || _instance$vis$data$se === void 0 ? void 0 : _instance$vis$data$se.getSerializedFields()) || {};
        instance.vis.setState({
          ...visState,
          data: {
            aggs,
            searchSource: {
              ...visSearchSource,
              query,
              filter
            },
            savedSearchId: instance.vis.data.savedSearchId
          }
        }).then(() => {
          // setting up the stateContainer after setState is successful will prevent loading the editor with failures
          // otherwise the catch will take presedence
          setAppState(stateContainer);
        }).catch(error => {
          // if setting new vis state was failed for any reason,
          // redirect to the listing page with error message
          services.toastNotifications.addWarning({
            title: _i18n.i18n.translate('visualizations.visualizationLoadingFailedErrorMessage', {
              defaultMessage: 'Failed to load the visualization'
            }),
            text: (0, _public.toMountPoint)( /*#__PURE__*/_react.default.createElement(_public.KibanaThemeProvider, {
              theme$: services.theme.theme$
            }, /*#__PURE__*/_react.default.createElement(_public.MarkdownSimple, null, error.message)))
          });
          services.history.replace(`${_constants.VisualizeConstants.LANDING_PAGE_PATH}?notFound=visualization`);
        });
      } else {
        setAppState(stateContainer);
      }

      // don't forget to clean up
      return () => {
        eventEmitter.off('dirtyStateChange', onDirtyStateChange);
        stopStateSync();
        stopSyncingAppFilters();
      };
    }
  }, [eventEmitter, instance, services]);
  return {
    appState,
    hasUnappliedChanges
  };
};
exports.useVisualizeAppState = useVisualizeAppState;