"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useSavedVisInstance = void 0;
var _react = require("react");
var _queryString = require("query-string");
var _i18n = require("@kbn/i18n");
var _get_visualization_instance = require("../get_visualization_instance");
var _breadcrumbs = require("../breadcrumbs");
var _constants = require("../../../../common/constants");
var _services = require("../../../services");
var _utils = require("../utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

/**
 * This effect is responsible for instantiating a saved vis or creating a new one
 * using url parameters, embedding and destroying it in DOM
 */
const useSavedVisInstance = (services, eventEmitter, isChromeVisible, originatingApp, visualizationIdFromUrl, embeddableInput) => {
  const [state, setState] = (0, _react.useState)({});
  const visEditorRef = (0, _react.useRef)(null);
  const visId = (0, _react.useRef)('');
  (0, _react.useEffect)(() => {
    var _state$savedVisInstan;
    const {
      chrome,
      history,
      toastNotifications,
      stateTransferService,
      visEditorsRegistry,
      application: {
        navigateToApp
      }
    } = services;
    const getSavedVisInstance = async () => {
      try {
        let savedVisInstance;
        if (history.location.pathname === '/create') {
          const searchParams = (0, _queryString.parse)(history.location.search);
          const visType = (0, _services.getTypes)().all().find(({
            name
          }) => name === searchParams.type);
          if (!visType) {
            throw new Error(_i18n.i18n.translate('visualizations.createVisualization.noVisTypeErrorMessage', {
              defaultMessage: 'You must provide a valid visualization type'
            }));
          }
          const shouldHaveIndex = visType.requiresSearch && visType.options.showIndexSelection;
          const hasIndex = searchParams.indexPattern || searchParams.savedSearchId;
          if (shouldHaveIndex && !hasIndex) {
            throw new Error(_i18n.i18n.translate('visualizations.createVisualization.noIndexPatternOrSavedSearchIdErrorMessage', {
              defaultMessage: 'You must provide either an indexPattern or a savedSearchId'
            }));
          }
          savedVisInstance = await (0, _get_visualization_instance.getVisualizationInstance)(services, searchParams);
        } else {
          savedVisInstance = await (0, _get_visualization_instance.getVisualizationInstance)(services, visualizationIdFromUrl);
        }
        if (embeddableInput && embeddableInput.timeRange) {
          savedVisInstance.panelTimeRange = embeddableInput.timeRange;
        }
        const {
          embeddableHandler,
          savedVis,
          vis
        } = savedVisInstance;
        const originatingAppName = originatingApp ? stateTransferService.getAppNameFromId(originatingApp) : undefined;
        const redirectToOrigin = originatingApp ? () => navigateToApp(originatingApp) : undefined;
        if (savedVis.id) {
          chrome.setBreadcrumbs((0, _breadcrumbs.getEditBreadcrumbs)({
            originatingAppName,
            redirectToOrigin
          }, savedVis.title));
          chrome.docTitle.change(savedVis.title);
        } else {
          chrome.setBreadcrumbs((0, _breadcrumbs.getCreateBreadcrumbs)({
            byValue: Boolean(originatingApp),
            originatingAppName,
            redirectToOrigin
          }));
        }
        let visEditorController;
        // do not create editor in embeded mode
        if (visEditorRef.current) {
          if (isChromeVisible) {
            var _vis$type$editorConfi;
            const Editor = visEditorsRegistry.get((_vis$type$editorConfi = vis.type.editorConfig) === null || _vis$type$editorConfi === void 0 ? void 0 : _vis$type$editorConfi.editor);
            if (Editor) {
              visEditorController = new Editor(visEditorRef.current, vis, eventEmitter, embeddableHandler);
            }
          } else {
            embeddableHandler.render(visEditorRef.current);
          }
        }
        setState({
          savedVisInstance,
          visEditorController
        });
      } catch (error) {
        try {
          (0, _utils.redirectToSavedObjectPage)(services, error, visualizationIdFromUrl);
        } catch (e) {
          toastNotifications.addWarning({
            title: _i18n.i18n.translate('visualizations.createVisualization.failedToLoadErrorMessage', {
              defaultMessage: 'Failed to load the visualization'
            }),
            text: e.message
          });
          history.replace(_constants.VisualizeConstants.LANDING_PAGE_PATH);
        }
      }
    };
    if (isChromeVisible === undefined) {
      // wait for specifying chrome
      return;
    }
    if (!visId.current) {
      visId.current = visualizationIdFromUrl || 'new';
      getSavedVisInstance();
    } else if (visualizationIdFromUrl && visId.current !== visualizationIdFromUrl && ((_state$savedVisInstan = state.savedVisInstance) === null || _state$savedVisInstan === void 0 ? void 0 : _state$savedVisInstan.savedVis.id) !== visualizationIdFromUrl) {
      visId.current = visualizationIdFromUrl;
      setState({});
      getSavedVisInstance();
    }
  }, [services, eventEmitter, originatingApp, isChromeVisible, visualizationIdFromUrl, state.savedVisInstance, state.visEditorController, embeddableInput]);
  (0, _react.useEffect)(() => {
    return () => {
      var _state$savedVisInstan2;
      if (state.visEditorController) {
        state.visEditorController.destroy();
      } else if ((_state$savedVisInstan2 = state.savedVisInstance) !== null && _state$savedVisInstan2 !== void 0 && _state$savedVisInstan2.embeddableHandler) {
        state.savedVisInstance.embeddableHandler.destroy();
      }
    };
  }, [state]);
  return {
    ...state,
    visEditorRef
  };
};
exports.useSavedVisInstance = useSavedVisInstance;