"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = exports.Vis = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _lodash = require("lodash");
var _i18n = require("@kbn/i18n");
var _public = require("@kbn/data-views-plugin/public");
var _public2 = require("@kbn/saved-search-plugin/public");
var _persisted_state = require("./persisted_state");
var _services = require("./services");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

/**
 * @name Vis
 *
 * @description This class consists of aggs, params, listeners, title, and type.
 *  - Aggs: Instances of IAggConfig.
 *  - Params: The settings in the Options tab.
 *
 * Not to be confused with vislib/vis.js.
 */

const getSearchSource = async (inputSearchSource, savedSearchId) => {
  if (savedSearchId) {
    var _savedSearch;
    let savedSearch;
    try {
      var _getSavedObjectTaggin;
      savedSearch = await (0, _public2.getSavedSearch)(savedSearchId, {
        search: (0, _services.getSearch)(),
        savedObjectsClient: (0, _services.getSavedObjects)().client,
        spaces: (0, _services.getSpaces)(),
        savedObjectsTagging: (_getSavedObjectTaggin = (0, _services.getSavedObjectTagging)()) === null || _getSavedObjectTaggin === void 0 ? void 0 : _getSavedObjectTaggin.getTaggingApi()
      });
    } catch (e) {
      return inputSearchSource;
    }
    await (0, _public2.throwErrorOnSavedSearchUrlConflict)(savedSearch);
    if ((_savedSearch = savedSearch) !== null && _savedSearch !== void 0 && _savedSearch.searchSource) {
      inputSearchSource.setParent(savedSearch.searchSource);
    }
  }
  return inputSearchSource;
};
class Vis {
  constructor(visType, visState = {}) {
    (0, _defineProperty2.default)(this, "type", void 0);
    (0, _defineProperty2.default)(this, "id", void 0);
    (0, _defineProperty2.default)(this, "title", '');
    (0, _defineProperty2.default)(this, "description", '');
    (0, _defineProperty2.default)(this, "params", void 0);
    (0, _defineProperty2.default)(this, "data", {});
    (0, _defineProperty2.default)(this, "uiState", void 0);
    this.type = this.getType(visType);
    this.params = this.getParams(visState.params);
    this.uiState = new _persisted_state.PersistedState(visState.uiState);
    this.id = visState.id;
  }
  getType(visType) {
    const type = (0, _services.getTypes)().get(visType);
    if (!type) {
      const errorMessage = _i18n.i18n.translate('visualizations.visualizationTypeInvalidMessage', {
        defaultMessage: 'Invalid visualization type "{visType}"',
        values: {
          visType
        }
      });
      throw new Error(errorMessage);
    }
    return type;
  }
  getParams(params) {
    var _this$type$visConfig$, _this$type$visConfig;
    return (0, _lodash.defaults)({}, (0, _lodash.cloneDeep)(params !== null && params !== void 0 ? params : {}), (0, _lodash.cloneDeep)((_this$type$visConfig$ = (_this$type$visConfig = this.type.visConfig) === null || _this$type$visConfig === void 0 ? void 0 : _this$type$visConfig.defaults) !== null && _this$type$visConfig$ !== void 0 ? _this$type$visConfig$ : {}));
  }
  async setState(inState) {
    let state = inState;
    const {
      updateVisTypeOnParamsChange
    } = this.type;
    const newType = updateVisTypeOnParamsChange && updateVisTypeOnParamsChange(state.params);
    if (newType) {
      state = {
        ...inState,
        type: newType,
        params: {
          ...inState.params,
          type: newType
        }
      };
    }
    let typeChanged = false;
    if (state.type && this.type.name !== state.type) {
      // @ts-ignore
      this.type = this.getType(state.type);
      typeChanged = true;
    }
    if (state.title !== undefined) {
      this.title = state.title;
    }
    if (state.description !== undefined) {
      this.description = state.description;
    }
    if (state.params || typeChanged) {
      this.params = this.getParams(state.params);
    }
    try {
      if (state.data && state.data.searchSource) {
        this.data.searchSource = await (0, _services.getSearch)().searchSource.create(state.data.searchSource);
        this.data.indexPattern = this.data.searchSource.getField('index');
      }
    } catch (e) {
      // nothing to be here
    }
    try {
      if (state.data && state.data.savedSearchId) {
        if (this.data.searchSource) {
          this.data.searchSource = await getSearchSource(this.data.searchSource, state.data.savedSearchId);
          this.data.indexPattern = this.data.searchSource.getField('index');
        }
        this.data.savedSearchId = state.data.savedSearchId;
      }
    } catch (e) {
      // nothing to be here
    }
    if (state.data && (state.data.aggs || !this.data.aggs)) {
      const aggs = state.data.aggs ? (0, _lodash.cloneDeep)(state.data.aggs) : [];
      const configStates = this.initializeDefaultsFromSchemas(aggs, this.type.schemas.all || []);
      if (!this.data.indexPattern && aggs.length) {
        var _state$data$searchSou, _state$data$searchSou2, _state$data$searchSou3, _state$data$searchSou4, _state$data$savedSear, _this$data$searchSour;
        const dataViewId = typeof ((_state$data$searchSou = state.data.searchSource) === null || _state$data$searchSou === void 0 ? void 0 : _state$data$searchSou.index) === 'string' ? (_state$data$searchSou2 = state.data.searchSource) === null || _state$data$searchSou2 === void 0 ? void 0 : _state$data$searchSou2.index : (_state$data$searchSou3 = state.data.searchSource) === null || _state$data$searchSou3 === void 0 ? void 0 : (_state$data$searchSou4 = _state$data$searchSou3.index) === null || _state$data$searchSou4 === void 0 ? void 0 : _state$data$searchSou4.id;
        this.data.indexPattern = new _public.DataView({
          spec: {
            id: (_state$data$savedSear = state.data.savedSearchId) !== null && _state$data$savedSear !== void 0 ? _state$data$savedSear : dataViewId
          },
          fieldFormats: (0, _services.getFieldsFormats)()
        });
        this.data.searchSource = await (0, _services.getSearch)().searchSource.createEmpty();
        (_this$data$searchSour = this.data.searchSource) === null || _this$data$searchSour === void 0 ? void 0 : _this$data$searchSour.setField('index', this.data.indexPattern);
      }
      if (this.data.indexPattern) {
        this.data.aggs = (0, _services.getAggs)().createAggConfigs(this.data.indexPattern, configStates);
      }
    }
  }
  clone() {
    const {
      data,
      ...restOfSerialized
    } = this.serialize();
    const vis = new Vis(this.type.name, restOfSerialized);
    vis.setState({
      ...restOfSerialized,
      data: {}
    });
    const aggs = this.data.indexPattern ? (0, _services.getAggs)().createAggConfigs(this.data.indexPattern, data.aggs) : undefined;
    vis.data = {
      ...this.data,
      aggs
    };
    return vis;
  }
  serialize() {
    const aggs = this.data.aggs ? this.data.aggs.aggs.map(agg => agg.serialize()) : [];
    return {
      id: this.id,
      title: this.title,
      description: this.description,
      type: this.type.name,
      params: (0, _lodash.cloneDeep)(this.params),
      uiState: this.uiState.toJSON(),
      data: {
        aggs: aggs,
        searchSource: this.data.searchSource ? this.data.searchSource.getSerializedFields() : {},
        ...(this.data.savedSearchId ? {
          savedSearchId: this.data.savedSearchId
        } : {})
      }
    };
  }

  // deprecated
  isHierarchical() {
    if ((0, _lodash.isFunction)(this.type.hierarchicalData)) {
      return !!this.type.hierarchicalData(this);
    } else {
      return !!this.type.hierarchicalData;
    }
  }
  initializeDefaultsFromSchemas(configStates, schemas) {
    // Set the defaults for any schema which has them. If the defaults
    // for some reason has more then the max only set the max number
    // of defaults (not sure why a someone define more...
    // but whatever). Also if a schema.name is already set then don't
    // set anything.
    const newConfigs = [...configStates];
    schemas.filter(schema => Array.isArray(schema.defaults) && schema.defaults.length > 0).filter(schema => !configStates.find(agg => agg.schema && agg.schema === schema.name)).forEach(schema => {
      const defaultSchemaConfig = schema.defaults.slice(0, schema.max);
      defaultSchemaConfig.forEach(d => newConfigs.push(d));
    });
    return newConfigs;
  }
}

// eslint-disable-next-line import/no-default-export
exports.Vis = Vis;
var _default = Vis;
exports.default = _default;