"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.convertToLens = void 0;
var _uuid = require("uuid");
var _common = require("@kbn/data-plugin/common");
var _convert_to_lens = require("@kbn/visualizations-plugin/common/convert_to_lens");
var _enums = require("../../../common/enums");
var _services = require("../../services");
var _datasource = require("../lib/datasource");
var _series = require("../lib/series");
var _metric = require("../lib/configurations/metric");
var _metrics = require("../lib/metrics");
var _utils = require("../utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

const MAX_SERIES = 2;
const MAX_BUCKETS = 2;
const invalidModelError = () => new Error('Invalid model');
const convertToLens = async ({
  params: model
}, timeRange) => {
  const dataViews = (0, _services.getDataViewsStart)();
  try {
    const seriesNum = model.series.filter(series => !series.hidden).length;
    const indexPatternIds = new Set();
    // we should get max only 2 series
    const visibleSeries = model.series.filter(({
      hidden
    }) => !hidden).slice(0, 2);
    let currentIndexPattern = null;
    for (const series of visibleSeries) {
      const datasourceInfo = await (0, _datasource.extractOrGenerateDatasourceInfo)(model.index_pattern, model.time_field, Boolean(series.override_index_pattern), series.series_index_pattern, series.series_time_field, dataViews);
      if (!datasourceInfo) {
        throw invalidModelError();
      }
      const {
        indexPatternId,
        indexPattern
      } = datasourceInfo;
      indexPatternIds.add(indexPatternId);
      currentIndexPattern = indexPattern;
    }
    if (indexPatternIds.size > 1) {
      throw invalidModelError();
    }
    const [indexPatternId] = indexPatternIds.values();
    const buckets = [];
    const metrics = [];

    // handle multiple layers/series
    for (const series of visibleSeries) {
      // not valid time shift
      if (series.offset_time && (0, _common.parseTimeShift)(series.offset_time) === 'invalid') {
        throw invalidModelError();
      }
      if (!(0, _metrics.isValidMetrics)(series.metrics, _enums.PANEL_TYPES.METRIC, series.time_range_mode)) {
        throw invalidModelError();
      }
      const reducedTimeRange = (0, _metrics.getReducedTimeRange)(model, series, timeRange);

      // handle multiple metrics
      const metricsColumns = (0, _series.getMetricsColumns)(series, currentIndexPattern, seriesNum, {
        reducedTimeRange
      });
      if (metricsColumns === null) {
        throw invalidModelError();
      }
      const bucketsColumns = (0, _series.getBucketsColumns)(model, series, metricsColumns, currentIndexPattern, false);
      if (bucketsColumns === null) {
        throw invalidModelError();
      }
      buckets.push(...bucketsColumns);
      metrics.push(...metricsColumns);
    }
    let uniqueBuckets = buckets;
    if (visibleSeries.length === MAX_SERIES && buckets.length) {
      if (buckets.length !== MAX_BUCKETS) {
        throw invalidModelError();
      }
      uniqueBuckets = (0, _utils.getUniqueBuckets)(buckets);
      if (uniqueBuckets.length !== 1) {
        throw invalidModelError();
      }
    }
    const [bucket] = uniqueBuckets;
    const extendedLayer = {
      indexPatternId: indexPatternId,
      layerId: (0, _uuid.v4)(),
      columns: [...metrics, ...(bucket ? [bucket] : [])],
      columnOrder: []
    };
    const configuration = (0, _metric.getConfigurationForMetric)(model, extendedLayer, bucket);
    if (!configuration) {
      throw invalidModelError();
    }
    const layers = Object.values((0, _utils.excludeMetaFromLayers)({
      0: extendedLayer
    }));
    return {
      type: 'lnsMetric',
      layers,
      configuration,
      indexPatternIds: (0, _convert_to_lens.getIndexPatternIds)(layers)
    };
  } catch (e) {
    return null;
  }
};
exports.convertToLens = convertToLens;