"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.convertToLens = void 0;
var _uuid = require("uuid");
var _common = require("@kbn/data-plugin/common");
var _convert_to_lens = require("@kbn/visualizations-plugin/common/convert_to_lens");
var _enums = require("../../../common/enums");
var _services = require("../../services");
var _datasource = require("../lib/datasource");
var _series = require("../lib/series");
var _metric = require("../lib/configurations/metric");
var _metrics = require("../lib/metrics");
var _convert = require("../lib/convert");
var _utils = require("../utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

const getMaxFormula = (metric, column) => {
  var _metric$field;
  const baseFormula = `overall_max`;
  if (column && column.operationType === 'formula') {
    return `${baseFormula}(${column.params.formula})`;
  }
  return `${baseFormula}(${(0, _metrics.getFormulaFromMetric)(_metrics.SUPPORTED_METRICS[metric.type])}(${(_metric$field = metric.field) !== null && _metric$field !== void 0 ? _metric$field : ''}))`;
};
const invalidModelError = () => new Error('Invalid model');
const convertToLens = async ({
  params: model
}, timeRange) => {
  const dataViews = (0, _services.getDataViewsStart)();
  try {
    var _gaugeMaxColumn;
    const series = model.series[0];
    // not valid time shift
    if (series.offset_time && (0, _common.parseTimeShift)(series.offset_time) === 'invalid') {
      throw invalidModelError();
    }
    if (!(0, _metrics.isValidMetrics)(series.metrics, _enums.PANEL_TYPES.GAUGE, series.time_range_mode)) {
      throw invalidModelError();
    }
    if (series.metrics[series.metrics.length - 1].type === _enums.TSVB_METRIC_TYPES.STATIC) {
      throw invalidModelError();
    }
    const reducedTimeRange = (0, _metrics.getReducedTimeRange)(model, series, timeRange);
    const datasourceInfo = await (0, _datasource.extractOrGenerateDatasourceInfo)(model.index_pattern, model.time_field, Boolean(series.override_index_pattern), series.series_index_pattern, series.series_time_field, dataViews);
    if (!datasourceInfo) {
      throw invalidModelError();
    }
    const {
      indexPatternId,
      indexPattern
    } = datasourceInfo;

    // handle multiple metrics
    const metricsColumns = (0, _series.getMetricsColumns)(series, indexPattern, model.series.length, {
      reducedTimeRange
    });
    if (metricsColumns === null) {
      throw invalidModelError();
    }
    const bucketsColumns = (0, _series.getBucketsColumns)(model, series, metricsColumns, indexPattern, false);
    if (bucketsColumns === null) {
      throw invalidModelError();
    }
    const [bucket] = bucketsColumns;
    const extendedLayer = {
      indexPatternId,
      layerId: (0, _uuid.v4)(),
      columns: [...metricsColumns, ...(bucket ? [bucket] : [])],
      columnOrder: []
    };
    const primarySeries = model.series[0];
    const primaryMetricWithCollapseFn = (0, _utils.getMetricWithCollapseFn)(primarySeries);
    if (!primaryMetricWithCollapseFn || !primaryMetricWithCollapseFn.metric) {
      throw invalidModelError();
    }
    const primaryColumn = (0, _utils.findMetricColumn)(primaryMetricWithCollapseFn.metric, extendedLayer.columns);
    if (!primaryColumn) {
      throw invalidModelError();
    }
    let gaugeMaxColumn = (0, _convert.createFormulaColumnWithoutMeta)(getMaxFormula(primaryMetricWithCollapseFn.metric, primaryColumn));
    if (model.gauge_max !== undefined && model.gauge_max !== '') {
      gaugeMaxColumn = (0, _convert.createStaticValueColumn)(model.gauge_max);
    }
    const layer = {
      ...extendedLayer,
      columns: [...extendedLayer.columns, gaugeMaxColumn]
    };
    const configuration = (0, _metric.getConfigurationForGauge)(model, layer, bucket, (_gaugeMaxColumn = gaugeMaxColumn) !== null && _gaugeMaxColumn !== void 0 ? _gaugeMaxColumn : undefined);
    if (!configuration) {
      throw invalidModelError();
    }
    const layers = Object.values((0, _utils.excludeMetaFromLayers)({
      0: layer
    }));
    return {
      type: 'lnsMetric',
      layers,
      configuration,
      indexPatternIds: (0, _convert_to_lens.getIndexPatternIds)(layers)
    };
  } catch (e) {
    return null;
  }
};
exports.convertToLens = convertToLens;