"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.convertSeriesToDataTable = exports.addMetaToColumns = void 0;
var _common = require("@kbn/data-plugin/common");
var _fields_utils = require("../../../../common/fields_utils");
var _enums = require("../../../../common/enums");
var _index_patterns_utils = require("../../../../common/index_patterns_utils");
var _services = require("../../../services");
var _constants = require("../../visualizations/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

const addMetaToColumns = (columns, indexPattern) => {
  return columns.map(column => {
    const field = indexPattern.getFieldByName(column.name);
    const type = (field === null || field === void 0 ? void 0 : field.spec.type) || 'number';
    let params = {
      field: field === null || field === void 0 ? void 0 : field.spec.name
    };
    if (column.type === _common.BUCKET_TYPES.MULTI_TERMS) {
      params = {
        fields: column.fields,
        otherBucket: true
      };
    } else if (column.type === _enums.BUCKET_TYPES.FILTERS && column.params) {
      const filters = column.params.map(col => ({
        input: col.filter,
        label: col.label
      }));
      params = {
        filters
      };
    } else if (column.type === 'date_histogram') {
      const {
        query
      } = (0, _services.getDataStart)();
      const timeRange = query.timefilter.timefilter.getTime();
      params = {
        timeRange
      };
    }
    const cleanedColumn = {
      id: column.id.toString(),
      name: column.name,
      meta: {
        type,
        field: field === null || field === void 0 ? void 0 : field.spec.name,
        index: indexPattern.title,
        source: 'esaggs',
        sourceParams: {
          enabled: true,
          indexPatternId: indexPattern === null || indexPattern === void 0 ? void 0 : indexPattern.id,
          type: column.type,
          schema: column.isMetric ? 'metric' : 'group',
          params
        }
      }
    };
    return cleanedColumn;
  });
};
exports.addMetaToColumns = addMetaToColumns;
const hasSeriesAgg = metrics => {
  return metrics.some(metric => metric.type === _enums.TSVB_METRIC_TYPES.SERIES_AGG);
};
const convertSeriesToDataTable = async (model, series, initialIndexPattern) => {
  const tables = {};
  const dataViews = (0, _services.getDataViewsStart)();
  for (let layerIdx = 0; layerIdx < model.series.length; layerIdx++) {
    const layer = model.series[layerIdx];
    let usedIndexPattern = initialIndexPattern;
    // The user can overwrite the index pattern of a layer.
    // In that case, the index pattern should be fetched again.
    if (layer.override_index_pattern) {
      const {
        indexPattern
      } = await (0, _index_patterns_utils.fetchIndexPattern)(layer.series_index_pattern, dataViews);
      if (indexPattern) {
        usedIndexPattern = indexPattern;
      }
    }
    // series aggregation is a special case, splitting by terms doesn't create multiple series per term
    const isGroupedByTerms = layer.split_mode === _enums.BUCKET_TYPES.TERMS && !hasSeriesAgg(layer.metrics);
    const isGroupedByFilters = layer.split_mode === _enums.BUCKET_TYPES.FILTERS;
    const seriesPerLayer = series.filter(s => s.seriesId === layer.id);
    let id = _constants.X_ACCESSOR_INDEX;
    const columns = [{
      id,
      name: usedIndexPattern.timeFieldName || '',
      isMetric: false,
      type: 'date_histogram'
    }];
    if (seriesPerLayer.length) {
      id++;
      const metrics = layer.metrics;
      columns.push({
        id,
        name: metrics[metrics.length - 1].field || seriesPerLayer[0].splitByLabel,
        isMetric: true,
        type: metrics[metrics.length - 1].type
      });

      // Adds an extra column, if the layer is split by terms or filters aggregation
      if (isGroupedByTerms) {
        const fieldsForTerms = (0, _fields_utils.getFieldsForTerms)(layer.terms_field);
        id++;
        columns.push({
          id,
          name: (0, _fields_utils.getMultiFieldLabel)(fieldsForTerms),
          fields: fieldsForTerms,
          isMetric: false,
          type: fieldsForTerms.length > 1 ? _common.BUCKET_TYPES.MULTI_TERMS : _enums.BUCKET_TYPES.TERMS
        });
      } else if (isGroupedByFilters) {
        id++;
        columns.push({
          id,
          name: _enums.BUCKET_TYPES.FILTERS,
          isMetric: false,
          params: layer === null || layer === void 0 ? void 0 : layer.split_filters,
          type: _enums.BUCKET_TYPES.FILTERS
        });
      }
    }
    const columnsWithMeta = addMetaToColumns(columns, usedIndexPattern);
    const filtersColumn = columns.find(col => col.type === _enums.BUCKET_TYPES.FILTERS);
    let rows = [];
    for (let j = 0; j < seriesPerLayer.length; j++) {
      const {
        data,
        label,
        isSplitByTerms,
        termsSplitKey
      } = seriesPerLayer[j];
      const seriesData = data.map(rowData => {
        let rowId = _constants.X_ACCESSOR_INDEX;
        const rowsData = {
          [rowId++]: rowData[0],
          [rowId++]: rowData[1]
        };
        let splitValue;
        if (isGroupedByTerms || filtersColumn) {
          const termsValue = Array.isArray(termsSplitKey) ? new _common.MultiFieldKey({
            key: termsSplitKey
          }) : termsSplitKey;
          splitValue = {
            [rowId]: isSplitByTerms && termsValue !== undefined ? termsValue : [label].flat()[0]
          };
        }
        return splitValue ? {
          ...rowsData,
          ...splitValue
        } : rowsData;
      });
      rows = [...rows, ...seriesData];
    }
    tables[layer.id] = {
      type: 'datatable',
      rows,
      columns: columnsWithMeta
    };
  }
  return tables;
};
exports.convertSeriesToDataTable = convertSeriesToDataTable;