"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useSavedQuery = void 0;
var _react = require("react");
var _i18n = require("@kbn/i18n");
var _populate_state_from_saved_query = require("./populate_state_from_saved_query");
var _clear_saved_query = require("./clear_saved_query");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

const useSavedQuery = props => {
  // Handle saved queries
  const [savedQuery, setSavedQuery] = (0, _react.useState)();

  // Effect is used to convert a saved query id into an object
  (0, _react.useEffect)(() => {
    const fetchSavedQuery = async savedQueryId => {
      try {
        // fetch saved query
        const newSavedQuery = await props.queryService.savedQueries.getSavedQuery(savedQueryId);
        // Make sure we set the saved query to the most recent one
        if (newSavedQuery && newSavedQuery.id === savedQueryId) {
          setSavedQuery(newSavedQuery);
          (0, _populate_state_from_saved_query.populateStateFromSavedQuery)(props.queryService, newSavedQuery);
        }
      } catch (error) {
        // Clear saved query
        setSavedQuery(undefined);
        (0, _clear_saved_query.clearStateFromSavedQuery)(props.queryService);
        // notify of saving error
        props.notifications.toasts.addWarning({
          title: _i18n.i18n.translate('unifiedSearch.search.unableToGetSavedQueryToastTitle', {
            defaultMessage: 'Unable to load query {savedQueryId}',
            values: {
              savedQueryId
            }
          }),
          text: `${error.message}`
        });
      }
    };
    if (props.savedQueryId) fetchSavedQuery(props.savedQueryId);else setSavedQuery(undefined);
  }, [props.notifications.toasts, props.queryService, props.queryService.savedQueries, props.savedQueryId]);
  return {
    savedQuery,
    setSavedQuery: q => {
      setSavedQuery(q);
      (0, _populate_state_from_saved_query.populateStateFromSavedQuery)(props.queryService, q);
    },
    clearSavedQuery: () => {
      setSavedQuery(undefined);
      (0, _clear_saved_query.clearStateFromSavedQuery)(props.queryService);
    }
  };
};
exports.useSavedQuery = useSavedQuery;