"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useGroupedFields = useGroupedFields;
var _lodash = require("lodash");
var _react = require("react");
var _i18n = require("@kbn/i18n");
var _types = require("../types");
var _use_existing_fields = require("./use_existing_fields");
var _use_field_filters = require("./use_field_filters");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

function useGroupedFields({
  dataViewId,
  allFields,
  services,
  isAffectedByGlobalFilter = false,
  popularFieldsLimit,
  sortedSelectedFields,
  getCustomFieldType,
  onOverrideFieldGroupDetails,
  onSupportedFieldFilter,
  onSelectedFieldFilter
}) {
  const fieldsExistenceReader = (0, _use_existing_fields.useExistingFieldsReader)();
  const fieldListFilters = (0, _use_field_filters.useFieldFilters)({
    allFields,
    services,
    getCustomFieldType,
    onSupportedFieldFilter
  });
  const onFilterFieldList = fieldListFilters.onFilterField;
  const [dataView, setDataView] = (0, _react.useState)(null);
  const isAffectedByTimeFilter = Boolean(dataView === null || dataView === void 0 ? void 0 : dataView.timeFieldName);
  const fieldsExistenceInfoUnavailable = dataViewId ? fieldsExistenceReader.isFieldsExistenceInfoUnavailable(dataViewId) : true;
  const hasFieldDataHandler = dataViewId ? fieldsExistenceReader.hasFieldData : hasFieldDataByDefault;
  (0, _react.useEffect)(() => {
    const getDataView = async () => {
      if (dataViewId) {
        let nextDataView = null;
        try {
          nextDataView = await services.dataViews.get(dataViewId, false);
        } catch (e) {
          //
        }
        setDataView(nextDataView || null);
      } else {
        setDataView(null);
      }
    };
    getDataView();
    // if field existence information changed, reload the data view too
  }, [dataViewId, services.dataViews, setDataView, hasFieldDataHandler]);

  // important when switching from a known dataViewId to no data view (like in text-based queries)
  (0, _react.useEffect)(() => {
    if (dataView && !dataViewId) {
      setDataView(null);
    }
  }, [dataView, setDataView, dataViewId]);
  const scrollToTopResetCounter = (0, _react.useMemo)(() => Date.now(),
  // eslint-disable-next-line react-hooks/exhaustive-deps
  [dataViewId, onFilterFieldList]);
  const unfilteredFieldGroups = (0, _react.useMemo)(() => {
    const containsData = field => {
      return dataViewId ? hasFieldDataHandler(dataViewId, field.name) : true;
    };
    const selectedFields = sortedSelectedFields || [];
    const sortedFields = [...(allFields || [])].sort(sortFields);
    const groupedFields = {
      ...getDefaultFieldGroups(),
      ...(0, _lodash.groupBy)(sortedFields, field => {
        var _dataView$metaFields;
        if (!sortedSelectedFields && onSelectedFieldFilter && onSelectedFieldFilter(field)) {
          selectedFields.push(field);
        }
        if (onSupportedFieldFilter && !onSupportedFieldFilter(field)) {
          return 'skippedFields';
        }
        if (field.type === 'document') {
          return 'specialFields';
        }
        if (dataView !== null && dataView !== void 0 && (_dataView$metaFields = dataView.metaFields) !== null && _dataView$metaFields !== void 0 && _dataView$metaFields.includes(field.name)) {
          return 'metaFields';
        }
        // `nested` root fields are not a part of data view fields list, so we need to check them separately
        if (field.type === 'nested') {
          return 'availableFields';
        }
        if (dataView !== null && dataView !== void 0 && dataView.getFieldByName && !dataView.getFieldByName(field.name)) {
          return 'unmappedFields';
        }
        if (containsData(field) || fieldsExistenceInfoUnavailable) {
          return 'availableFields';
        }
        return 'emptyFields';
      })
    };
    const popularFields = popularFieldsLimit ? sortedFields.filter(field => field.count && field.type !== '_source' && (!onSupportedFieldFilter || onSupportedFieldFilter(field))).sort((a, b) => (b.count || 0) - (a.count || 0)) // sort by popularity score
    .slice(0, popularFieldsLimit) : [];
    let fieldGroupDefinitions = {
      SpecialFields: {
        fields: groupedFields.specialFields,
        fieldCount: groupedFields.specialFields.length,
        isAffectedByGlobalFilter: false,
        isAffectedByTimeFilter: false,
        isInitiallyOpen: false,
        showInAccordion: false,
        title: '',
        hideDetails: true
      },
      SelectedFields: {
        fields: selectedFields,
        fieldCount: selectedFields.length,
        isInitiallyOpen: true,
        showInAccordion: true,
        title: _i18n.i18n.translate('unifiedFieldList.useGroupedFields.selectedFieldsLabel', {
          defaultMessage: 'Selected fields'
        }),
        isAffectedByGlobalFilter,
        isAffectedByTimeFilter,
        hideDetails: false,
        hideIfEmpty: true
      },
      PopularFields: {
        fields: popularFields,
        fieldCount: popularFields.length,
        isInitiallyOpen: true,
        showInAccordion: true,
        title: _i18n.i18n.translate('unifiedFieldList.useGroupedFields.popularFieldsLabel', {
          defaultMessage: 'Popular fields'
        }),
        helpText: _i18n.i18n.translate('unifiedFieldList.useGroupedFields.popularFieldsLabelHelp', {
          defaultMessage: 'Fields that your organization frequently uses, from most to least popular.'
        }),
        isAffectedByGlobalFilter,
        isAffectedByTimeFilter,
        hideDetails: false,
        hideIfEmpty: true
      },
      AvailableFields: {
        fields: groupedFields.availableFields,
        fieldCount: groupedFields.availableFields.length,
        isInitiallyOpen: true,
        showInAccordion: true,
        title: dataViewId && fieldsExistenceInfoUnavailable ? _i18n.i18n.translate('unifiedFieldList.useGroupedFields.allFieldsLabel', {
          defaultMessage: 'All fields'
        }) : _i18n.i18n.translate('unifiedFieldList.useGroupedFields.availableFieldsLabel', {
          defaultMessage: 'Available fields'
        }),
        isAffectedByGlobalFilter,
        isAffectedByTimeFilter,
        // Show details on timeout but not failure
        // hideDetails: fieldsExistenceInfoUnavailable && !existenceFetchTimeout, // TODO: is this check still necessary?
        hideDetails: fieldsExistenceInfoUnavailable,
        defaultNoFieldsMessage: _i18n.i18n.translate('unifiedFieldList.useGroupedFields.noAvailableDataLabel', {
          defaultMessage: `There are no available fields that contain data.`
        })
      },
      UnmappedFields: {
        fields: groupedFields.unmappedFields,
        fieldCount: groupedFields.unmappedFields.length,
        isAffectedByGlobalFilter,
        isAffectedByTimeFilter,
        isInitiallyOpen: false,
        showInAccordion: true,
        hideDetails: false,
        hideIfEmpty: true,
        title: _i18n.i18n.translate('unifiedFieldList.useGroupedFields.unmappedFieldsLabel', {
          defaultMessage: 'Unmapped fields'
        }),
        helpText: _i18n.i18n.translate('unifiedFieldList.useGroupedFields.unmappedFieldsLabelHelp', {
          defaultMessage: "Fields that aren't explicitly mapped to a field data type."
        })
      },
      EmptyFields: {
        fields: groupedFields.emptyFields,
        fieldCount: groupedFields.emptyFields.length,
        isAffectedByGlobalFilter: false,
        isAffectedByTimeFilter: false,
        isInitiallyOpen: false,
        showInAccordion: true,
        hideDetails: false,
        hideIfEmpty: !dataViewId,
        title: _i18n.i18n.translate('unifiedFieldList.useGroupedFields.emptyFieldsLabel', {
          defaultMessage: 'Empty fields'
        }),
        helpText: _i18n.i18n.translate('unifiedFieldList.useGroupedFields.emptyFieldsLabelHelp', {
          defaultMessage: "Fields that don't have any values based on your filters."
        }),
        defaultNoFieldsMessage: _i18n.i18n.translate('unifiedFieldList.useGroupedFields.noEmptyDataLabel', {
          defaultMessage: `There are no empty fields.`
        })
      },
      MetaFields: {
        fields: groupedFields.metaFields,
        fieldCount: groupedFields.metaFields.length,
        isAffectedByGlobalFilter: false,
        isAffectedByTimeFilter: false,
        isInitiallyOpen: false,
        showInAccordion: true,
        hideDetails: false,
        hideIfEmpty: !dataViewId,
        title: _i18n.i18n.translate('unifiedFieldList.useGroupedFields.metaFieldsLabel', {
          defaultMessage: 'Meta fields'
        }),
        defaultNoFieldsMessage: _i18n.i18n.translate('unifiedFieldList.useGroupedFields.noMetaDataLabel', {
          defaultMessage: `There are no meta fields.`
        })
      }
    };

    // do not show empty field accordion if there is no existence information
    if (fieldsExistenceInfoUnavailable) {
      delete fieldGroupDefinitions.EmptyFields;
    }
    if (onOverrideFieldGroupDetails) {
      fieldGroupDefinitions = Object.keys(fieldGroupDefinitions).reduce((definitions, name) => {
        const groupName = name;
        const group = fieldGroupDefinitions[groupName];
        if (group) {
          definitions[groupName] = {
            ...group,
            ...(onOverrideFieldGroupDetails(groupName) || {})
          };
        }
        return definitions;
      }, {});
    }
    return fieldGroupDefinitions;
  }, [allFields, onSupportedFieldFilter, onSelectedFieldFilter, onOverrideFieldGroupDetails, dataView, dataViewId, hasFieldDataHandler, fieldsExistenceInfoUnavailable, isAffectedByGlobalFilter, isAffectedByTimeFilter, popularFieldsLimit, sortedSelectedFields]);
  const fieldGroups = (0, _react.useMemo)(() => {
    if (!onFilterFieldList) {
      return unfilteredFieldGroups;
    }
    return Object.fromEntries(Object.entries(unfilteredFieldGroups).map(([name, group]) => [name, {
      ...group,
      fieldSearchHighlight: fieldListFilters.fieldSearchHighlight,
      fields: group.fields.filter(onFilterFieldList)
    }]));
  }, [unfilteredFieldGroups, onFilterFieldList, fieldListFilters.fieldSearchHighlight]);
  const hasDataLoaded = Boolean(allFields);
  const allFieldsLength = allFields === null || allFields === void 0 ? void 0 : allFields.length;
  const fieldsExistInIndex = (0, _react.useMemo)(() => {
    return dataViewId ? Boolean(allFieldsLength) : true;
  }, [dataViewId, allFieldsLength]);
  const fieldsExistenceStatus = (0, _react.useMemo)(() => {
    if (!hasDataLoaded) {
      return _types.ExistenceFetchStatus.unknown; // to show loading indicator in the list
    }

    if (!dataViewId || !fieldsExistenceReader) {
      // ex. for text-based queries
      return _types.ExistenceFetchStatus.succeeded;
    }
    return fieldsExistenceReader.getFieldsExistenceStatus(dataViewId);
  }, [dataViewId, hasDataLoaded, fieldsExistenceReader]);
  const screenReaderDescriptionId = fieldListFilters.fieldListFiltersProps.screenReaderDescriptionId;
  const fieldListGroupedProps = (0, _react.useMemo)(() => {
    return {
      fieldGroups,
      scrollToTopResetCounter,
      fieldsExistInIndex,
      fieldsExistenceStatus,
      screenReaderDescriptionId
    };
  }, [fieldGroups, scrollToTopResetCounter, fieldsExistInIndex, fieldsExistenceStatus, screenReaderDescriptionId]);
  return {
    fieldListGroupedProps,
    fieldListFiltersProps: fieldListFilters.fieldListFiltersProps
  };
}
const collator = new Intl.Collator(undefined, {
  sensitivity: 'base'
});
function sortFields(fieldA, fieldB) {
  return collator.compare(fieldA.displayName || fieldA.name, fieldB.displayName || fieldB.name);
}
function hasFieldDataByDefault() {
  return true;
}
function getDefaultFieldGroups() {
  return {
    specialFields: [],
    availableFields: [],
    emptyFields: [],
    metaFields: [],
    unmappedFields: [],
    skippedFields: []
  };
}