"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.buildFieldList = buildFieldList;
exports.existingFields = existingFields;
exports.fetchFieldExistence = fetchFieldExistence;
exports.isBoomError = isBoomError;
exports.legacyExistingFields = legacyExistingFields;
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

/**
 * The number of docs to sample to determine field empty status.
 */

async function fetchFieldExistence({
  search,
  dataViewsService,
  dataView,
  dslQuery = {
    match_all: {}
  },
  fromDate,
  toDate,
  timeFieldName,
  includeFrozen,
  metaFields
}) {
  const allFields = buildFieldList(dataView, metaFields);
  const existingFieldList = await dataViewsService.getFieldsForIndexPattern(dataView, {
    // filled in by data views service
    pattern: '',
    indexFilter: toQuery(timeFieldName, fromDate, toDate, dslQuery)
  });
  return {
    indexPatternTitle: dataView.title,
    existingFieldNames: existingFields(existingFieldList, allFields)
  };
}

/**
 * Exported only for unit tests.
 */
function buildFieldList(indexPattern, metaFields) {
  return indexPattern.fields.map(field => {
    return {
      name: field.name,
      isScript: !!field.scripted,
      lang: field.lang,
      script: field.script,
      // id is a special case - it doesn't show up in the meta field list,
      // but as it's not part of source, it has to be handled separately.
      isMeta: (metaFields === null || metaFields === void 0 ? void 0 : metaFields.includes(field.name)) || field.name === '_id',
      runtimeField: !field.isMapped ? field.runtimeField : undefined
    };
  });
}
function toQuery(timeFieldName, fromDate, toDate, dslQuery) {
  const filter = timeFieldName && fromDate && toDate ? [{
    range: {
      [timeFieldName]: {
        format: 'strict_date_optional_time',
        gte: fromDate,
        lte: toDate
      }
    }
  }, dslQuery] : [dslQuery];
  const query = {
    bool: {
      filter
    }
  };
  return query;
}

/**
 * Exported only for unit tests.
 */
function existingFields(filteredFields, allFields) {
  const filteredFieldsSet = new Set(filteredFields.map(f => f.name));
  return allFields.filter(field => field.isScript || field.runtimeField || filteredFieldsSet.has(field.name)).map(f => f.name);
}

/**
 * Exported only for unit tests.
 */
function legacyExistingFields(docs, fields) {
  const missingFields = new Set(fields);
  for (const doc of docs) {
    if (missingFields.size === 0) {
      break;
    }
    missingFields.forEach(field => {
      let fieldStore = doc.fields;
      if (field.isMeta) {
        fieldStore = doc;
      }
      const value = fieldStore[field.name];
      if (Array.isArray(value) && value.length) {
        missingFields.delete(field);
      } else if (!Array.isArray(value) && value) {
        missingFields.delete(field);
      }
    });
  }
  return fields.filter(field => !missingFields.has(field)).map(f => f.name);
}
function isBoomError(error) {
  return error.isBoom === true;
}