"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useColumns = void 0;
var _react = require("react");
var _lodash = require("lodash");
var _toggle_column = require("./toggle_column");
var _use_fetch_browser_fields_capabilities = require("../use_fetch_browser_fields_capabilities");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const EMPTY_FIELDS = [{
  field: '*',
  include_unmapped: true
}];
const fieldTypeToDataGridColumnTypeMapper = fieldType => {
  if (fieldType === 'date') return 'datetime';
  if (fieldType === 'number') return 'numeric';
  if (fieldType === 'object') return 'json';
  return fieldType;
};
const getFieldCategoryFromColumnId = columnId => {
  const fieldName = columnId.split('.');
  if (fieldName.length === 1) {
    return 'base';
  }
  return fieldName[0];
};

/**
 * EUI Data Grid expects the columns to have a property 'schema' defined for proper sorting
 * this schema as its own types as can be check out in the docs so we add it here manually
 * https://eui.elastic.co/#/tabular-content/data-grid-schema-columns
 */
const euiColumnFactory = (columnId, browserFields, defaultColumns) => {
  const defaultColumn = getColumnByColumnId(defaultColumns, columnId);
  const column = defaultColumn ? defaultColumn : {
    id: columnId
  };
  const browserFieldsProps = getBrowserFieldProps(columnId, browserFields);
  return {
    ...column,
    schema: fieldTypeToDataGridColumnTypeMapper(browserFieldsProps.type)
  };
};
const getBrowserFieldProps = (columnId, browserFields) => {
  const notFoundSpecs = {
    type: 'string'
  };
  if (!browserFields || Object.keys(browserFields).length === 0) {
    return notFoundSpecs;
  }
  const category = getFieldCategoryFromColumnId(columnId);
  if (!browserFields[category]) {
    return notFoundSpecs;
  }
  const categorySpecs = browserFields[category].fields;
  if (!categorySpecs) {
    return notFoundSpecs;
  }
  const fieldSpecs = categorySpecs[columnId];
  return fieldSpecs ? fieldSpecs : notFoundSpecs;
};
const isPopulatedColumn = column => Boolean(column.schema);

/**
 * @param columns Columns to be considered in the alerts table
 * @param browserFields constant object with all field capabilities
 * @returns columns but with the info needed by the data grid to work as expected, e.g sorting
 */
const populateColumns = (columns, browserFields, defaultColumns) => {
  return columns.map(column => {
    return isPopulatedColumn(column) ? column : euiColumnFactory(column.id, browserFields, defaultColumns);
  });
};
const getColumnIds = columns => {
  return columns.map(column => column.id);
};
const getColumnByColumnId = (columns, columnId) => {
  return columns.find(({
    id
  }) => id === columnId);
};
const getColumnsByColumnIds = (columns, columnIds) => {
  return columnIds.map(columnId => columns.find(column => column.id === columnId)).filter(Boolean);
};
const persist = ({
  id,
  storageAlertsTable,
  columns,
  storage
}) => {
  storageAlertsTable.current = {
    ...storageAlertsTable.current,
    columns
  };
  storage.current.set(id, storageAlertsTable.current);
};
const useColumns = ({
  featureIds,
  storageAlertsTable,
  storage,
  id,
  defaultColumns,
  initialBrowserFields
}) => {
  const [isBrowserFieldDataLoading, browserFields] = (0, _use_fetch_browser_fields_capabilities.useFetchBrowserFieldCapabilities)({
    featureIds,
    initialBrowserFields
  });
  const [columns, setColumns] = (0, _react.useState)(() => {
    let cols = storageAlertsTable.current.columns;
    // before restoring from storage, enrich the column data
    if (initialBrowserFields && defaultColumns) {
      cols = populateColumns(cols, initialBrowserFields, defaultColumns);
    }
    return cols;
  });
  const [isColumnsPopulated, setColumnsPopulated] = (0, _react.useState)(false);
  const defaultColumnsRef = (0, _react.useRef)(defaultColumns);
  const didDefaultColumnChange = (0, _react.useMemo)(() => !(0, _lodash.isEqual)(defaultColumns, defaultColumnsRef.current), [defaultColumns]);
  (0, _react.useEffect)(() => {
    // if defaultColumns have changed,
    // get the latest columns provided by client and
    if (didDefaultColumnChange) {
      defaultColumnsRef.current = defaultColumns;
      setColumnsPopulated(false);
      setColumns(storageAlertsTable.current.columns);
      return;
    }
  }, [didDefaultColumnChange, storageAlertsTable, defaultColumns]);
  (0, _react.useEffect)(() => {
    if ((0, _lodash.isEmpty)(browserFields) || isColumnsPopulated) return;
    const populatedColumns = populateColumns(columns, browserFields, defaultColumns);
    setColumnsPopulated(true);
    setColumns(populatedColumns);
  }, [browserFields, defaultColumns, isBrowserFieldDataLoading, isColumnsPopulated, columns]);
  const setColumnsAndSave = (0, _react.useCallback)(newColumns => {
    setColumns(newColumns);
    persist({
      id,
      storage,
      storageAlertsTable,
      columns: newColumns
    });
  }, [id, storage, storageAlertsTable]);
  const setColumnsByColumnIds = (0, _react.useCallback)(columnIds => {
    const newColumns = getColumnsByColumnIds(columns, columnIds);
    setColumnsAndSave(newColumns);
  }, [setColumnsAndSave, columns]);
  const onToggleColumn = (0, _react.useCallback)(columnId => {
    const column = euiColumnFactory(columnId, browserFields, defaultColumns);
    const newColumns = (0, _toggle_column.toggleColumn)({
      column,
      columns,
      defaultColumns
    });
    setColumnsAndSave(newColumns);
  }, [browserFields, columns, defaultColumns, setColumnsAndSave]);
  const onResetColumns = (0, _react.useCallback)(() => {
    const populatedDefaultColumns = populateColumns(defaultColumns, browserFields, defaultColumns);
    setColumnsAndSave(populatedDefaultColumns);
  }, [browserFields, defaultColumns, setColumnsAndSave]);

  /*
   * In some case such security, we need some special fields such as threat.enrichments which are
   * not fetched when passing only EMPTY_FIELDS. Hence, we will fetch all the fields that user has added to the table.
   *
   * Additionaly, system such as o11y needs fields which are not even added in the table such as rule_type_id and hence we
   * additionly pass EMPTY_FIELDS so that it brings all fields apart from special fields
   *
   * */
  const fieldsToFetch = (0, _react.useMemo)(() => [...columns.map(col => ({
    field: col.id,
    include_unmapped: true
  })), ...EMPTY_FIELDS], [columns]);
  const visibleColumns = (0, _react.useMemo)(() => {
    return getColumnIds(columns);
  }, [columns]);
  return {
    columns,
    visibleColumns,
    isBrowserFieldDataLoading,
    browserFields,
    onColumnsChange: setColumnsAndSave,
    onToggleColumn,
    onResetColumns,
    onChangeVisibleColumns: setColumnsByColumnIds,
    fields: fieldsToFetch
  };
};
exports.useColumns = useColumns;