"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useEditTransformFlyout = exports.stringValidator = exports.retentionPolicyMaxAgeValidator = exports.integerRangeMinus1To100Validator = exports.integerRange10To10000Validator = exports.integerAboveZeroValidator = exports.initializeSection = exports.initializeField = exports.getDefaultState = exports.frequencyValidator = exports.formReducerFactory = exports.applyFormStateToTransformConfig = exports.EditTransformFlyoutProvider = exports.EDIT_TRANSFORM_HOOK_SELECTORS = void 0;
var _constate = _interopRequireDefault(require("constate"));
var _lodash = require("lodash");
var _react = require("react");
var _i18n = require("@kbn/i18n");
var _mlAggUtils = require("@kbn/ml-agg-utils");
var _mlNestedProperty = require("@kbn/ml-nested-property");
var _constants = require("../../../../../../common/constants");
var _validators = require("../../../../common/validators");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// Note on the form validation and input components used:
// All inputs use `EuiFieldText` which means all form values will be treated as strings.
// This means we cast other formats like numbers coming from the transform config to strings,
// then revalidate them and cast them again to number before submitting a transform update.
// We do this so we have fine grained control over field validation and the option to
// cast to special values like `null` for disabling `docs_per_second`.
const numberAboveZeroNotValidErrorMessage = _i18n.i18n.translate('xpack.transform.transformList.editFlyoutFormNumberAboveZeroNotValidErrorMessage', {
  defaultMessage: 'Value needs to be an integer above zero.'
});
const numberRangeMinus1To100NotValidErrorMessage = _i18n.i18n.translate('xpack.transform.transformList.editFlyoutFormNumberGreaterThanOrEqualToNegativeOneNotValidErrorMessage', {
  defaultMessage: 'Number of retries needs to be between 0 and 100, or -1 for infinite retries.'
});
const integerAboveZeroValidator = value => !(value + '').includes('.') && (0, _mlAggUtils.numberValidator)({
  min: 1,
  integerOnly: true
})(+value) === null ? [] : [numberAboveZeroNotValidErrorMessage];
exports.integerAboveZeroValidator = integerAboveZeroValidator;
const integerRangeMinus1To100Validator = value => !(value + '').includes('.') && (0, _mlAggUtils.numberValidator)({
  min: -1,
  max: 100,
  integerOnly: true
})(+value) === null ? [] : [numberRangeMinus1To100NotValidErrorMessage];
exports.integerRangeMinus1To100Validator = integerRangeMinus1To100Validator;
const numberRange10To10000NotValidErrorMessage = _i18n.i18n.translate('xpack.transform.transformList.editFlyoutFormNumberRange10To10000NotValidErrorMessage', {
  defaultMessage: 'Value needs to be an integer between 10 and 10000.'
});
const integerRange10To10000Validator = value => !(value + '').includes('.') && (0, _mlAggUtils.numberValidator)({
  min: 10,
  max: 100001,
  integerOnly: true
})(+value) === null ? [] : [numberRange10To10000NotValidErrorMessage];
exports.integerRange10To10000Validator = integerRange10To10000Validator;
const requiredErrorMessage = _i18n.i18n.translate('xpack.transform.transformList.editFlyoutFormRequiredErrorMessage', {
  defaultMessage: 'Required field.'
});
const stringNotValidErrorMessage = _i18n.i18n.translate('xpack.transform.transformList.editFlyoutFormStringNotValidErrorMessage', {
  defaultMessage: 'Value needs to be of type string.'
});
const stringValidator = (value, isOptional = true) => {
  if (typeof value !== 'string') {
    return [stringNotValidErrorMessage];
  }
  if (value.length === 0 && !isOptional) {
    return [requiredErrorMessage];
  }
  return [];
};
exports.stringValidator = stringValidator;
function parseDurationAboveZero(arg, errorMessage) {
  if (typeof arg !== 'string' || arg === null) {
    return [stringNotValidErrorMessage];
  }

  // split string by groups of numbers and letters
  const regexStr = arg.match(/[a-z]+|[^a-z]+/gi);

  // only valid if one group of numbers and one group of letters
  if (regexStr === null || Array.isArray(regexStr) && regexStr.length !== 2) {
    return [frequencyNotValidErrorMessage];
  }
  const number = +regexStr[0];
  const timeUnit = regexStr[1];

  // only valid if number is an integer above 0
  if (isNaN(number) || !Number.isInteger(number) || number === 0) {
    return [frequencyNotValidErrorMessage];
  }
  return {
    number,
    timeUnit
  };
}

// Only allow frequencies in the form of 1s/1h etc.
const frequencyNotValidErrorMessage = _i18n.i18n.translate('xpack.transform.transformList.editFlyoutFormFrequencyNotValidErrorMessage', {
  defaultMessage: 'The frequency value is not valid.'
});
const frequencyValidator = arg => {
  const parsedArg = parseDurationAboveZero(arg, frequencyNotValidErrorMessage);
  if (Array.isArray(parsedArg)) {
    return parsedArg;
  }
  return (0, _validators.isValidFrequency)(parsedArg) ? [] : [frequencyNotValidErrorMessage];
};

// Retention policy max age validator
exports.frequencyValidator = frequencyValidator;
const retentionPolicyMaxAgeNotValidErrorMessage = _i18n.i18n.translate('xpack.transform.transformList.editFlyoutFormRetentionPolicyMaxAgeNotValidErrorMessage', {
  defaultMessage: 'Invalid max age format. Minimum of 60s required.'
});
const retentionPolicyMaxAgeValidator = arg => {
  const parsedArg = parseDurationAboveZero(arg, retentionPolicyMaxAgeNotValidErrorMessage);
  if (Array.isArray(parsedArg)) {
    return parsedArg;
  }
  return (0, _validators.isValidRetentionPolicyMaxAge)(parsedArg) ? [] : [retentionPolicyMaxAgeNotValidErrorMessage];
};
exports.retentionPolicyMaxAgeValidator = retentionPolicyMaxAgeValidator;
const validate = {
  string: stringValidator,
  frequency: frequencyValidator,
  integerAboveZero: integerAboveZeroValidator,
  integerRangeMinus1To100: integerRangeMinus1To100Validator,
  integerRange10To10000: integerRange10To10000Validator,
  retentionPolicyMaxAge: retentionPolicyMaxAgeValidator
};
const initializeField = (formFieldName, configFieldName, config, overloads) => {
  const defaultValue = (overloads === null || overloads === void 0 ? void 0 : overloads.defaultValue) !== undefined ? overloads.defaultValue : '';
  const rawValue = (0, _mlNestedProperty.getNestedProperty)(config, configFieldName, undefined);
  const value = rawValue !== null && rawValue !== undefined ? rawValue.toString() : '';
  return {
    formFieldName,
    configFieldName,
    defaultValue,
    dependsOn: [],
    errorMessages: [],
    isNullable: false,
    isOptional: true,
    validator: 'string',
    value,
    valueParser: v => v,
    ...(overloads !== undefined ? {
      ...overloads
    } : {})
  };
};
exports.initializeField = initializeField;
const initializeSection = (formSectionName, configFieldName, config, overloads) => {
  var _overloads$defaultEna;
  const defaultEnabled = (_overloads$defaultEna = overloads === null || overloads === void 0 ? void 0 : overloads.defaultEnabled) !== null && _overloads$defaultEna !== void 0 ? _overloads$defaultEna : false;
  const rawEnabled = (0, _mlNestedProperty.getNestedProperty)(config, configFieldName, undefined);
  const enabled = rawEnabled !== undefined && rawEnabled !== null;
  return {
    formSectionName,
    configFieldName,
    defaultEnabled,
    enabled
  };
};
exports.initializeSection = initializeSection;
// Takes a value from form state and applies it to the structure
// of the expected final configuration request object.
// Considers options like if a value is nullable or optional.
const getUpdateValue = (attribute, config, formState, enforceFormValue = false) => {
  const {
    formFields,
    formSections
  } = formState;
  const formStateAttribute = formFields[attribute];
  const fallbackValue = formStateAttribute.isNullable ? null : formStateAttribute.defaultValue;
  const enabledBasedOnSection = formStateAttribute.section !== undefined ? formSections[formStateAttribute.section].enabled : true;
  const formValue = formStateAttribute.value !== '' ? formStateAttribute.valueParser(formStateAttribute.value) : fallbackValue;
  const configValue = (0, _mlNestedProperty.getNestedProperty)(config, formStateAttribute.configFieldName, fallbackValue);

  // only get depending values if we're not already in a call to get depending values.
  const dependsOnConfig = enforceFormValue === false ? formStateAttribute.dependsOn.reduce((_dependsOnConfig, dependsOnField) => {
    return (0, _lodash.merge)({
      ..._dependsOnConfig
    }, getUpdateValue(dependsOnField, config, formState, true));
  }, {}) : {};
  if (formValue === formStateAttribute.defaultValue && formValue === configValue && formStateAttribute.isOptional) {
    return {};
  }

  // If the resettable section the form field belongs to is disabled,
  // the whole section will be set to `null` to do the actual reset.
  if (formStateAttribute.section !== undefined && !enabledBasedOnSection) {
    return (0, _mlNestedProperty.setNestedProperty)(dependsOnConfig, formSections[formStateAttribute.section].configFieldName, null);
  }
  return enabledBasedOnSection && (formValue !== configValue || enforceFormValue) ? (0, _mlNestedProperty.setNestedProperty)(dependsOnConfig, formStateAttribute.configFieldName, formValue === '' && formStateAttribute.isOptional ? undefined : formValue) : {};
};

// Takes in the form configuration and returns a
// request object suitable to be sent to the
// transform update API endpoint.
const applyFormStateToTransformConfig = (config, formState) =>
// Iterates over all form fields and only if necessary applies them to
// the request object used for updating the transform.
Object.keys(formState.formFields).reduce((updateConfig, field) => (0, _lodash.merge)({
  ...updateConfig
}, getUpdateValue(field, config, formState)), {});

// Takes in a transform configuration and returns
// the default state to populate the form.
exports.applyFormStateToTransformConfig = applyFormStateToTransformConfig;
const getDefaultState = config => ({
  formFields: {
    // top level attributes
    description: initializeField('description', 'description', config),
    frequency: initializeField('frequency', 'frequency', config, {
      defaultValue: _constants.DEFAULT_TRANSFORM_FREQUENCY,
      validator: 'frequency'
    }),
    // dest.*
    destinationIndex: initializeField('destinationIndex', 'dest.index', config, {
      dependsOn: ['destinationIngestPipeline'],
      isOptional: false
    }),
    destinationIngestPipeline: initializeField('destinationIngestPipeline', 'dest.pipeline', config, {
      dependsOn: ['destinationIndex'],
      isOptional: true
    }),
    // settings.*
    docsPerSecond: initializeField('docsPerSecond', 'settings.docs_per_second', config, {
      isNullable: true,
      isOptional: true,
      validator: 'integerAboveZero',
      valueParser: v => v === '' ? null : +v
    }),
    maxPageSearchSize: initializeField('maxPageSearchSize', 'settings.max_page_search_size', config, {
      defaultValue: `${_constants.DEFAULT_TRANSFORM_SETTINGS_MAX_PAGE_SEARCH_SIZE}`,
      isNullable: true,
      isOptional: true,
      validator: 'integerRange10To10000',
      valueParser: v => +v
    }),
    numFailureRetries: initializeField('numFailureRetries', 'settings.num_failure_retries', config, {
      defaultValue: undefined,
      isNullable: true,
      isOptional: true,
      validator: 'integerRangeMinus1To100',
      valueParser: v => +v
    }),
    // retention_policy.*
    retentionPolicyField: initializeField('retentionPolicyField', 'retention_policy.time.field', config, {
      dependsOn: ['retentionPolicyMaxAge'],
      isNullable: false,
      isOptional: true,
      section: 'retentionPolicy'
    }),
    retentionPolicyMaxAge: initializeField('retentionPolicyMaxAge', 'retention_policy.time.max_age', config, {
      dependsOn: ['retentionPolicyField'],
      isNullable: false,
      isOptional: true,
      section: 'retentionPolicy',
      validator: 'retentionPolicyMaxAge'
    })
  },
  formSections: {
    retentionPolicy: initializeSection('retentionPolicy', 'retention_policy', config)
  },
  isFormTouched: false,
  isFormValid: true
});

// Checks each form field for error messages to return
// if the overall form is valid or not.
exports.getDefaultState = getDefaultState;
const isFormValid = fieldsState => Object.keys(fieldsState).reduce((p, c) => p && fieldsState[c].errorMessages.length === 0, true);

// Updates a form field with its new value,
// runs validation and populates
// `errorMessages` if any errors occur.
const formFieldReducer = (state, value) => {
  return {
    ...state,
    errorMessages: state.isOptional && typeof value === 'string' && value.length === 0 ? [] : validate[state.validator](value, state.isOptional),
    value
  };
};
const formSectionReducer = (state, enabled) => {
  return {
    ...state,
    enabled
  };
};
const getFieldValues = fields => Object.values(fields).map(f => f.value);
const getSectionValues = sections => Object.values(sections).map(s => s.enabled);

// Main form reducer triggers
// - `formFieldReducer` to update the actions field
// - compares the most recent state against the original one to update `isFormTouched`
// - sets `isFormValid` to have a flag if any of the form fields contains an error.
const formReducerFactory = config => {
  const defaultState = getDefaultState(config);
  const defaultFieldValues = getFieldValues(defaultState.formFields);
  const defaultSectionValues = getSectionValues(defaultState.formSections);
  return (state, action) => {
    const formFields = action.name === 'form_field' ? {
      ...state.formFields,
      [action.payload.field]: formFieldReducer(state.formFields[action.payload.field], action.payload.value)
    } : state.formFields;
    const formSections = action.name === 'form_section' ? {
      ...state.formSections,
      [action.payload.section]: formSectionReducer(state.formSections[action.payload.section], action.payload.enabled)
    } : state.formSections;
    return {
      ...state,
      apiErrorMessage: action.name === 'api_error' ? action.payload : state.apiErrorMessage,
      formFields,
      formSections,
      isFormTouched: !(0, _lodash.isEqual)(defaultFieldValues, getFieldValues(formFields)) || !(0, _lodash.isEqual)(defaultSectionValues, getSectionValues(formSections)),
      isFormValid: isFormValid(formFields)
    };
  };
};
exports.formReducerFactory = formReducerFactory;
const useEditTransformFlyoutInternal = ({
  config,
  dataViewId
}) => {
  const [formState, dispatch] = (0, _react.useReducer)(formReducerFactory(config), getDefaultState(config));
  const actions = (0, _react.useMemo)(() => ({
    apiError: payload => dispatch({
      name: 'api_error',
      payload
    }),
    formField: payload => dispatch({
      name: 'form_field',
      payload
    }),
    formSection: payload => dispatch({
      name: 'form_section',
      payload
    })
  }), []);
  const requestConfig = (0, _react.useMemo)(() => applyFormStateToTransformConfig(config, formState), [config, formState]);
  const isUpdateButtonDisabled = (0, _react.useMemo)(() => !formState.isFormValid || !formState.isFormTouched, [formState.isFormValid, formState.isFormTouched]);
  return {
    config,
    dataViewId,
    formState,
    actions,
    requestConfig,
    isUpdateButtonDisabled
  };
};

// wrap hook with the constate factory to create context provider and custom hooks based on selectors
const [EditTransformFlyoutProvider, ...editTransformHooks] = (0, _constate.default)(useEditTransformFlyoutInternal, d => d.config, d => d.dataViewId, d => d.actions, d => d.formState.apiErrorMessage, d => d.formState.formSections, d => d.formState.formFields.description, d => d.formState.formFields.destinationIndex, d => d.formState.formFields.docsPerSecond, d => d.formState.formFields.frequency, d => d.formState.formFields.destinationIngestPipeline, d => d.formState.formFields.maxPageSearchSize, d => d.formState.formFields.numFailureRetries, d => d.formState.formFields.retentionPolicyField, d => d.formState.formFields.retentionPolicyMaxAge, d => d.requestConfig, d => d.isUpdateButtonDisabled);
exports.EditTransformFlyoutProvider = EditTransformFlyoutProvider;
let EDIT_TRANSFORM_HOOK_SELECTORS;
exports.EDIT_TRANSFORM_HOOK_SELECTORS = EDIT_TRANSFORM_HOOK_SELECTORS;
(function (EDIT_TRANSFORM_HOOK_SELECTORS) {
  EDIT_TRANSFORM_HOOK_SELECTORS[EDIT_TRANSFORM_HOOK_SELECTORS["config"] = 0] = "config";
  EDIT_TRANSFORM_HOOK_SELECTORS[EDIT_TRANSFORM_HOOK_SELECTORS["dataViewId"] = 1] = "dataViewId";
  EDIT_TRANSFORM_HOOK_SELECTORS[EDIT_TRANSFORM_HOOK_SELECTORS["actions"] = 2] = "actions";
  EDIT_TRANSFORM_HOOK_SELECTORS[EDIT_TRANSFORM_HOOK_SELECTORS["apiErrorMessage"] = 3] = "apiErrorMessage";
  EDIT_TRANSFORM_HOOK_SELECTORS[EDIT_TRANSFORM_HOOK_SELECTORS["stateFormSection"] = 4] = "stateFormSection";
  EDIT_TRANSFORM_HOOK_SELECTORS[EDIT_TRANSFORM_HOOK_SELECTORS["description"] = 5] = "description";
  EDIT_TRANSFORM_HOOK_SELECTORS[EDIT_TRANSFORM_HOOK_SELECTORS["destinationIndex"] = 6] = "destinationIndex";
  EDIT_TRANSFORM_HOOK_SELECTORS[EDIT_TRANSFORM_HOOK_SELECTORS["docsPerSecond"] = 7] = "docsPerSecond";
  EDIT_TRANSFORM_HOOK_SELECTORS[EDIT_TRANSFORM_HOOK_SELECTORS["frequency"] = 8] = "frequency";
  EDIT_TRANSFORM_HOOK_SELECTORS[EDIT_TRANSFORM_HOOK_SELECTORS["destinationIngestPipeline"] = 9] = "destinationIngestPipeline";
  EDIT_TRANSFORM_HOOK_SELECTORS[EDIT_TRANSFORM_HOOK_SELECTORS["maxPageSearchSize"] = 10] = "maxPageSearchSize";
  EDIT_TRANSFORM_HOOK_SELECTORS[EDIT_TRANSFORM_HOOK_SELECTORS["numFailureRetries"] = 11] = "numFailureRetries";
  EDIT_TRANSFORM_HOOK_SELECTORS[EDIT_TRANSFORM_HOOK_SELECTORS["retentionPolicyField"] = 12] = "retentionPolicyField";
  EDIT_TRANSFORM_HOOK_SELECTORS[EDIT_TRANSFORM_HOOK_SELECTORS["retentionPolicyMaxAge"] = 13] = "retentionPolicyMaxAge";
  EDIT_TRANSFORM_HOOK_SELECTORS[EDIT_TRANSFORM_HOOK_SELECTORS["requestConfig"] = 14] = "requestConfig";
  EDIT_TRANSFORM_HOOK_SELECTORS[EDIT_TRANSFORM_HOOK_SELECTORS["isUpdateButtonDisabled"] = 15] = "isUpdateButtonDisabled";
})(EDIT_TRANSFORM_HOOK_SELECTORS || (exports.EDIT_TRANSFORM_HOOK_SELECTORS = EDIT_TRANSFORM_HOOK_SELECTORS = {}));
const useEditTransformFlyout = hookKey => {
  return editTransformHooks[EDIT_TRANSFORM_HOOK_SELECTORS[hookKey]]();
};
exports.useEditTransformFlyout = useEditTransformFlyout;