"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.TRANSFORM_ERROR_TYPE = exports.REFRESH_TRANSFORM_LIST_STATE = void 0;
exports.isTransformIdValid = isTransformIdValid;
exports.useRefreshTransformList = exports.refreshTransformList$ = exports.overrideTransformForCloning = void 0;
var _react = require("react");
var _rxjs = require("rxjs");
var _operators = require("rxjs/operators");
var _lodash = require("lodash");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// Via https://github.com/elastic/elasticsearch/blob/master/x-pack/plugin/core/src/main/java/org/elasticsearch/xpack/core/transform/utils/TransformStrings.java#L24
// Matches a string that contains lowercase characters, digits, hyphens, underscores or dots.
// The string may start and end only in characters or digits.
// Note that '.' is allowed but not documented.
function isTransformIdValid(transformId) {
  return /^[a-z0-9](?:[a-z0-9_\-\.]*[a-z0-9])?$/g.test(transformId);
}
const TRANSFORM_ERROR_TYPE = {
  DANGLING_TASK: 'dangling_task'
};
exports.TRANSFORM_ERROR_TYPE = TRANSFORM_ERROR_TYPE;
let REFRESH_TRANSFORM_LIST_STATE;
exports.REFRESH_TRANSFORM_LIST_STATE = REFRESH_TRANSFORM_LIST_STATE;
(function (REFRESH_TRANSFORM_LIST_STATE) {
  REFRESH_TRANSFORM_LIST_STATE["ERROR"] = "error";
  REFRESH_TRANSFORM_LIST_STATE["IDLE"] = "idle";
  REFRESH_TRANSFORM_LIST_STATE["LOADING"] = "loading";
  REFRESH_TRANSFORM_LIST_STATE["REFRESH"] = "refresh";
})(REFRESH_TRANSFORM_LIST_STATE || (exports.REFRESH_TRANSFORM_LIST_STATE = REFRESH_TRANSFORM_LIST_STATE = {}));
const refreshTransformList$ = new _rxjs.BehaviorSubject(REFRESH_TRANSFORM_LIST_STATE.IDLE);
exports.refreshTransformList$ = refreshTransformList$;
const useRefreshTransformList = (callback = {}) => {
  (0, _react.useEffect)(() => {
    const distinct$ = refreshTransformList$.pipe((0, _operators.distinctUntilChanged)());
    const subscriptions = [];
    if (typeof callback.onRefresh === 'function') {
      // initial call to refresh
      callback.onRefresh();
      subscriptions.push(distinct$.pipe((0, _operators.filter)(state => state === REFRESH_TRANSFORM_LIST_STATE.REFRESH)).subscribe(() => typeof callback.onRefresh === 'function' && callback.onRefresh()));
    }
    if (typeof callback.isLoading === 'function') {
      subscriptions.push(distinct$.subscribe(state => typeof callback.isLoading === 'function' && callback.isLoading(state === REFRESH_TRANSFORM_LIST_STATE.LOADING)));
    }
    return () => {
      subscriptions.map(sub => sub.unsubscribe());
    };
    // The effect should only be called once.
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, []);
  return {
    refresh: () => {
      // A refresh is followed immediately by setting the state to loading
      // to trigger data fetching and loading indicators in one go.
      refreshTransformList$.next(REFRESH_TRANSFORM_LIST_STATE.REFRESH);
      refreshTransformList$.next(REFRESH_TRANSFORM_LIST_STATE.LOADING);
    }
  };
};
exports.useRefreshTransformList = useRefreshTransformList;
const overrideTransformForCloning = originalConfig => {
  var _clonedConfig$_meta;
  // 'Managed' means job is preconfigured and deployed by other solutions
  // we should not clone this setting
  const clonedConfig = (0, _lodash.cloneDeep)(originalConfig);
  (_clonedConfig$_meta = clonedConfig._meta) === null || _clonedConfig$_meta === void 0 ? true : delete _clonedConfig$_meta.managed;
  return clonedConfig;
};
exports.overrideTransformForCloning = overrideTransformForCloning;