"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.INITIAL_CAPABILITIES = void 0;
exports.createCapabilityFailureMessage = createCapabilityFailureMessage;
exports.toArray = exports.hasPrivilegeFactory = exports.getPrivilegesAndCapabilities = exports.extractMissingPrivileges = void 0;
var _i18n = require("@kbn/i18n");
var _mlIsPopulatedObject = require("@kbn/ml-is-populated-object");
var _lodash = require("lodash");
var _constants = require("../constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const INITIAL_CAPABILITIES = Object.freeze({
  canGetTransform: false,
  canDeleteTransform: false,
  canPreviewTransform: false,
  canCreateTransform: false,
  canReauthorizeTransform: false,
  canScheduleNowTransform: false,
  canStartStopTransform: false,
  canCreateTransformAlerts: false,
  canUseTransformAlerts: false,
  canResetTransform: false
});
exports.INITIAL_CAPABILITIES = INITIAL_CAPABILITIES;
function isPrivileges(arg) {
  return (0, _mlIsPopulatedObject.isPopulatedObject)(arg, ['hasAllPrivileges', 'missingPrivileges']) && typeof arg.hasAllPrivileges === 'boolean' && typeof arg.missingPrivileges === 'object' && arg.missingPrivileges !== null;
}
const toArray = value => Array.isArray(value) ? value : [value];
exports.toArray = toArray;
const hasPrivilegeFactory = privileges => privilege => {
  const [section, requiredPrivilege] = privilege;
  if (isPrivileges(privileges) && !privileges.missingPrivileges[section]) {
    // if the section does not exist in our missingPrivileges, everything is OK
    return true;
  }
  if (isPrivileges(privileges) && privileges.missingPrivileges[section].length === 0) {
    return true;
  }
  if (requiredPrivilege === '*') {
    // If length > 0 and we require them all... KO
    return false;
  }
  // If we require _some_ privilege, we make sure that the one
  // we require is *not* in the missingPrivilege array
  return isPrivileges(privileges) && !privileges.missingPrivileges[section].includes(requiredPrivilege);
};
exports.hasPrivilegeFactory = hasPrivilegeFactory;
const extractMissingPrivileges = (privilegesObject = {}) => Object.keys(privilegesObject).reduce((privileges, privilegeName) => {
  if (!privilegesObject[privilegeName]) {
    privileges.push(privilegeName);
  }
  return privileges;
}, []);
exports.extractMissingPrivileges = extractMissingPrivileges;
const getPrivilegesAndCapabilities = (clusterPrivileges, hasOneIndexWithAllPrivileges, hasAllPrivileges) => {
  const privilegesResult = {
    hasAllPrivileges: true,
    missingPrivileges: {
      cluster: [],
      index: []
    }
  };

  // Find missing cluster privileges and set overall app privileges
  privilegesResult.missingPrivileges.cluster = extractMissingPrivileges(clusterPrivileges);
  privilegesResult.hasAllPrivileges = hasAllPrivileges;
  if (!hasOneIndexWithAllPrivileges) {
    privilegesResult.missingPrivileges.index = [..._constants.APP_INDEX_PRIVILEGES];
  }
  const hasPrivilege = hasPrivilegeFactory(privilegesResult);
  const capabilities = (0, _lodash.cloneDeep)(INITIAL_CAPABILITIES);
  capabilities.canGetTransform = hasPrivilege(['cluster', 'cluster:monitor/transform/get']) && hasPrivilege(['cluster', 'cluster:monitor/transform/stats/get']);
  capabilities.canCreateTransform = hasPrivilege(['cluster', 'cluster:admin/transform/put']);
  capabilities.canDeleteTransform = hasPrivilege(['cluster', 'cluster:admin/transform/delete']);
  capabilities.canResetTransform = hasPrivilege(['cluster', 'cluster:admin/transform/reset']);
  capabilities.canPreviewTransform = hasPrivilege(['cluster', 'cluster:admin/transform/preview']);
  capabilities.canStartStopTransform = hasPrivilege(['cluster', 'cluster:admin/transform/start']) && hasPrivilege(['cluster', 'cluster:admin/transform/start_task']) && hasPrivilege(['cluster', 'cluster:admin/transform/stop']);
  capabilities.canCreateTransformAlerts = capabilities.canCreateTransform;
  capabilities.canUseTransformAlerts = capabilities.canGetTransform;
  capabilities.canScheduleNowTransform = capabilities.canStartStopTransform;
  capabilities.canReauthorizeTransform = capabilities.canStartStopTransform;
  return {
    privileges: privilegesResult,
    capabilities
  };
};
// create the text for button's tooltips if the user
// doesn't have the permission to press that button
exports.getPrivilegesAndCapabilities = getPrivilegesAndCapabilities;
function createCapabilityFailureMessage(capability) {
  let message = '';
  switch (capability) {
    case 'canCreateTransform':
      message = _i18n.i18n.translate('xpack.transform.capability.noPermission.createTransformTooltip', {
        defaultMessage: 'You do not have permission to create transforms.'
      });
      break;
    case 'canCreateTransformAlerts':
      message = _i18n.i18n.translate('xpack.transform.capability.noPermission.canCreateTransformAlertsTooltip', {
        defaultMessage: 'You do not have permission to create transform alert rules.'
      });
      break;
    case 'canScheduleNowTransform':
      message = _i18n.i18n.translate('xpack.transform.capability.noPermission.scheduleNowTransformTooltip', {
        defaultMessage: 'You do not have permission to schedule transforms to process data instantly.'
      });
      break;
    case 'canStartStopTransform':
      message = _i18n.i18n.translate('xpack.transform.capability.noPermission.startOrStopTransformTooltip', {
        defaultMessage: 'You do not have permission to start or stop transforms.'
      });
      break;
    case 'canReauthorizeTransform':
      message = _i18n.i18n.translate('xpack.transform.capability.noPermission.reauthorizeTransformTooltip', {
        defaultMessage: 'You do not have permission to reauthorize transforms.'
      });
      break;
    case 'canDeleteTransform':
      message = _i18n.i18n.translate('xpack.transform.capability.noPermission.deleteTransformTooltip', {
        defaultMessage: 'You do not have permission to delete transforms.'
      });
      break;
    case 'canResetTransform':
      message = _i18n.i18n.translate('xpack.transform.capability.noPermission.resetTransformTooltip', {
        defaultMessage: 'You do not have permission to reset transforms.'
      });
      break;
    case 'noTransformNodes':
      message = _i18n.i18n.translate('xpack.transform.capability.noPermission.noTransformNodesTooltip', {
        defaultMessage: 'There are no transform nodes available.'
      });
      break;
  }
  return _i18n.i18n.translate('xpack.transform.capability.pleaseContactAdministratorTooltip', {
    defaultMessage: '{message} Please contact your administrator.',
    values: {
      message
    }
  });
}