"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.updateFiltersArray = exports.FilterOut = exports.FilterIn = void 0;
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const FilterIn = true;
exports.FilterIn = FilterIn;
const FilterOut = false;

/**
 * Creates a new filter to apply to the KQL bar.
 * @param key A string value mainly representing the field of an indicator
 * @param value A string value mainly representing the value of the indicator for the field
 * @param negate Set to true when we create a negated filter (e.g. NOT threat.indicator.type: url)
 * @returns The new {@link Filter}
 */
exports.FilterOut = FilterOut;
const createFilter = (key, value, negate) => ({
  meta: {
    alias: null,
    negate,
    disabled: false,
    type: 'phrase',
    key,
    params: {
      query: value
    }
  },
  query: {
    match_phrase: {
      [key]: value
    }
  }
});

/**
 * Checks if the key/value pair already exists in an array of filters.
 * @param filters Array of {@link Filter} retrieved from the SearchBar filterManager.
 * @param key A string value mainly representing the field of an indicator
 * @param value A string value mainly representing the value of the indicator for the field
 * @returns The new {@link Filter}
 */
const filterExistsInFiltersArray = (filters, key, value) => filters.find(f => {
  var _f$meta$params;
  return f.meta.key === key && typeof f.meta.params === 'object' && 'query' in f.meta.params && ((_f$meta$params = f.meta.params) === null || _f$meta$params === void 0 ? void 0 : _f$meta$params.query) === value;
});

/**
 * Returns true if the filter exists and should be removed, false otherwise (depending on a FilterIn or FilterOut action)
 * @param filter The {@link Filter}
 * @param filterType The type of action ({@link FilterIn} or {@link FilterOut})
 */
const shouldRemoveFilter = (filter, filterType) => filter != null && filterType === filter.meta.negate;

/**
 * Takes an array of filters and returns the updated array according to:
 * - if a filter already exists but negated, replace it by it's negation
 * - add the newly created filter
 * @param existingFilters List of {@link Filter} retrieved from the filterManager
 * @param key The value used in the newly created [@link Filter} as a key
 * @param value The value used in the newly created [@link Filter} as a params query
 * @param filterType Weather the function is called for a {@link FilterIn} or {@link FilterOut} action
 * @returns the updated array of filters
 */
const updateFiltersArray = (existingFilters, key, value, filterType) => {
  const newFilter = createFilter(key, value, !filterType);
  const filter = filterExistsInFiltersArray(existingFilters, key, value);
  return shouldRemoveFilter(filter, filterType) ? [...existingFilters.filter(f => f !== filter), newFilter] : [...existingFilters, newFilter];
};
exports.updateFiltersArray = updateFiltersArray;