"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.PollingErrorType = exports.PollingError = void 0;
exports.createTaskPoller = createTaskPoller;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _rxjs = require("rxjs");
var _operators = require("rxjs/operators");
var _Option = require("fp-ts/lib/Option");
var _result_type = require("../lib/result_type");
var _timeout_promise_after = require("./timeout_promise_after");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/*
 * This module contains the logic for polling the task manager index for new work.
 */

/**
 * constructs a new TaskPoller stream, which emits events on demand and on a scheduled interval, waiting for capacity to be available before emitting more events.
 *
 * @param opts
 * @prop {number} pollInterval - How often, in milliseconds, we will an event be emnitted, assuming there's capacity to do so
 * @prop {() => number} getCapacity - A function specifying whether there is capacity to emit new events
 * @prop {() => Promise<H>} work - The worker we wish to execute in order to `poll`
 *
 * @returns {Observable<Set<T>>} - An observable which emits an event whenever a polling event is due to take place, providing access to a singleton Set representing a queue
 *  of unique request argumets of type T.
 */
function createTaskPoller({
  logger,
  pollInterval$,
  pollIntervalDelay$,
  getCapacity,
  work,
  workTimeout
}) {
  const hasCapacity = () => getCapacity() > 0;
  const requestWorkProcessing$ = (0, _rxjs.combineLatest)([
  // emit a polling event on a fixed interval
  pollInterval$.pipe((0, _operators.tap)(period => {
    logger.debug(`Task poller now using interval of ${period}ms`);
  })), pollIntervalDelay$.pipe((0, _operators.tap)(pollDelay => {
    logger.debug(`Task poller now delaying emission by ${pollDelay}ms`);
  }))]).pipe(
  // We don't have control over `pollDelay` in the poller, and a change to `delayOnClaimConflicts` could accidentally cause us to pause Task Manager
  // polling for a far longer duration that we intended.
  // Since the goal is to shift it within the range of `period`, we use modulo as a safe guard to ensure this doesn't happen.
  (0, _operators.switchMap)(([period, pollDelay]) => (0, _rxjs.timer)(period + pollDelay % period, period)), (0, _operators.map)(() => _Option.none)).pipe(
  // only emit polling events when there's capacity to handle them
  (0, _operators.filter)(hasCapacity),
  // Run a cycle to poll for work
  (0, _operators.concatMap)(async () => {
    return (0, _result_type.map)(await (0, _result_type.promiseResult)((0, _timeout_promise_after.timeoutPromiseAfter)(work(), workTimeout, () => new Error(`work has timed out`))), workResult => (0, _result_type.asOk)(workResult), err => asPollingError(err, PollingErrorType.WorkError));
  }),
  // catch errors during polling for work
  (0, _operators.catchError)(err => (0, _rxjs.of)(asPollingError(err, PollingErrorType.WorkError))));
  return requestWorkProcessing$;
}
let PollingErrorType;
exports.PollingErrorType = PollingErrorType;
(function (PollingErrorType) {
  PollingErrorType[PollingErrorType["WorkError"] = 0] = "WorkError";
  PollingErrorType[PollingErrorType["WorkTimeout"] = 1] = "WorkTimeout";
  PollingErrorType[PollingErrorType["RequestCapacityReached"] = 2] = "RequestCapacityReached";
})(PollingErrorType || (exports.PollingErrorType = PollingErrorType = {}));
function asPollingError(err, type, data = _Option.none) {
  return (0, _result_type.asErr)(new PollingError(`Failed to poll for work: ${err}`, type, data));
}
class PollingError extends Error {
  constructor(message, type, data) {
    super(message);
    (0, _defineProperty2.default)(this, "type", void 0);
    (0, _defineProperty2.default)(this, "data", void 0);
    Object.setPrototypeOf(this, new.target.prototype);
    this.type = type;
    this.data = data;
  }
}
exports.PollingError = PollingError;