"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SyntheticsMonitorClient = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _utils = require("../utils");
var _synthetics_private_location = require("../private_location/synthetics_private_location");
var _format_configs = require("../formatters/format_configs");
var _synthetics_monitor = require("../../legacy_uptime/lib/saved_objects/synthetics_monitor");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class SyntheticsMonitorClient {
  constructor(syntheticsService, server) {
    (0, _defineProperty2.default)(this, "syntheticsService", void 0);
    (0, _defineProperty2.default)(this, "privateLocationAPI", void 0);
    this.syntheticsService = syntheticsService;
    this.privateLocationAPI = new _synthetics_private_location.SyntheticsPrivateLocation(server);
  }
  async addMonitors(monitors, request, savedObjectsClient, allPrivateLocations, spaceId) {
    const privateConfigs = [];
    const publicConfigs = [];
    const paramsBySpace = await this.syntheticsService.getSyntheticsParams({
      spaceId
    });
    for (const monitorObj of monitors) {
      const {
        monitor,
        id
      } = monitorObj;
      const config = {
        monitor,
        configId: id,
        params: paramsBySpace[spaceId]
      };
      const {
        str: paramsString,
        params
      } = (0, _format_configs.mixParamsWithGlobalParams)(paramsBySpace[spaceId], monitor);
      const formattedConfig = (0, _format_configs.formatHeartbeatRequest)(config, paramsString);
      const {
        privateLocations,
        publicLocations
      } = this.parseLocations(formattedConfig);
      if (privateLocations.length > 0) {
        privateConfigs.push({
          config: formattedConfig,
          globalParams: params
        });
      }
      if (publicLocations.length > 0) {
        publicConfigs.push(config);
      }
    }
    const newPolicies = this.privateLocationAPI.createPackagePolicies(privateConfigs, request, savedObjectsClient, allPrivateLocations, spaceId);
    const syncErrors = this.syntheticsService.addConfigs(publicConfigs);
    return await Promise.all([newPolicies, syncErrors]);
  }
  async editMonitors(monitors, routeContext, allPrivateLocations, spaceId) {
    const {
      request,
      savedObjectsClient
    } = routeContext;
    const privateConfigs = [];
    const publicConfigs = [];
    const deletedPublicConfigs = [];
    const paramsBySpace = await this.syntheticsService.getSyntheticsParams({
      spaceId
    });
    for (const editedMonitor of monitors) {
      const {
        str: paramsString,
        params
      } = (0, _format_configs.mixParamsWithGlobalParams)(paramsBySpace[spaceId], editedMonitor.monitor);
      const configData = {
        params: paramsBySpace[spaceId],
        monitor: editedMonitor.monitor,
        configId: editedMonitor.id
      };
      const editedConfig = (0, _format_configs.formatHeartbeatRequest)(configData, paramsString);
      const {
        publicLocations,
        privateLocations
      } = this.parseLocations(editedConfig);
      if (publicLocations.length > 0) {
        publicConfigs.push(configData);
      }
      const deletedPublicConfig = this.hasDeletedPublicLocations(publicLocations, editedMonitor.decryptedPreviousMonitor);
      if (deletedPublicConfig) {
        deletedPublicConfigs.push({
          ...deletedPublicConfig,
          params: paramsBySpace[spaceId]
        });
      }
      if (privateLocations.length > 0 || this.hasPrivateLocations(editedMonitor.previousMonitor)) {
        privateConfigs.push({
          config: editedConfig,
          globalParams: params
        });
      }
    }
    if (deletedPublicConfigs.length > 0) {
      await this.syntheticsService.deleteConfigs(deletedPublicConfigs);
    }
    const privateEditPromise = this.privateLocationAPI.editMonitors(privateConfigs, request, savedObjectsClient, allPrivateLocations, spaceId);
    const publicConfigsPromise = this.syntheticsService.editConfig(publicConfigs);
    const [publicSyncErrors, privateEditResponse] = await Promise.all([publicConfigsPromise, privateEditPromise]);
    const {
      failedUpdates: failedPolicyUpdates
    } = privateEditResponse;
    return {
      failedPolicyUpdates,
      publicSyncErrors
    };
  }
  async deleteMonitors(monitors, request, savedObjectsClient, spaceId) {
    const privateDeletePromise = this.privateLocationAPI.deleteMonitors(monitors, request, savedObjectsClient, spaceId);
    const publicDeletePromise = this.syntheticsService.deleteConfigs(monitors.map(monitor => ({
      monitor,
      configId: monitor.config_id,
      params: {}
    })));
    const [pubicResponse] = await Promise.all([publicDeletePromise, privateDeletePromise]);
    return pubicResponse;
  }
  hasPrivateLocations(previousMonitor) {
    const {
      locations
    } = previousMonitor.attributes;
    return locations.some(loc => !loc.isServiceManaged);
  }
  hasDeletedPublicLocations(updatedLocations, decryptedPreviousMonitor) {
    const {
      locations
    } = decryptedPreviousMonitor.attributes;
    const prevPublicLocations = locations.filter(loc => loc.isServiceManaged);
    const missingPublicLocations = prevPublicLocations.filter(prevLoc => {
      return !updatedLocations.some(updatedLoc => updatedLoc.id === prevLoc.id);
    });
    if (missingPublicLocations.length > 0) {
      const {
        attributes: normalizedPreviousMonitor
      } = (0, _utils.normalizeSecrets)(decryptedPreviousMonitor);
      normalizedPreviousMonitor.locations = missingPublicLocations;
      return {
        monitor: normalizedPreviousMonitor,
        configId: decryptedPreviousMonitor.id
      };
    }
  }
  parseLocations(config) {
    const {
      locations
    } = config;
    const privateLocations = locations.filter(loc => !loc.isServiceManaged);
    const publicLocations = locations.filter(loc => loc.isServiceManaged);
    return {
      privateLocations,
      publicLocations
    };
  }
  async syncGlobalParams({
    request,
    spaceId,
    savedObjectsClient,
    allPrivateLocations,
    encryptedSavedObjects
  }) {
    const privateConfigs = [];
    const publicConfigs = [];
    const {
      allConfigs: monitors,
      paramsBySpace
    } = await this.getAllMonitorConfigs({
      encryptedSavedObjects,
      spaceId
    });
    for (const monitor of monitors) {
      const {
        publicLocations,
        privateLocations
      } = this.parseLocations(monitor);
      if (publicLocations.length > 0) {
        publicConfigs.push({
          monitor,
          configId: monitor.config_id,
          params: {}
        });
      }
      if (privateLocations.length > 0) {
        privateConfigs.push({
          config: monitor,
          globalParams: paramsBySpace[monitor.namespace]
        });
      }
    }
    if (privateConfigs.length > 0) {
      await this.privateLocationAPI.editMonitors(privateConfigs, request, savedObjectsClient, allPrivateLocations, spaceId);
    }
    if (publicConfigs.length > 0) {
      return await this.syntheticsService.editConfig(publicConfigs, false);
    }
  }
  async getAllMonitorConfigs({
    spaceId,
    encryptedSavedObjects
  }) {
    const paramsBySpacePromise = this.syntheticsService.getSyntheticsParams({
      spaceId
    });
    const monitorsPromise = this.getAllMonitors({
      encryptedSavedObjects,
      spaceId
    });
    const [paramsBySpace, monitors] = await Promise.all([paramsBySpacePromise, monitorsPromise]);
    return {
      allConfigs: this.mixParamsWithMonitors(spaceId, monitors, paramsBySpace),
      paramsBySpace
    };
  }
  async getAllMonitors({
    spaceId,
    encryptedSavedObjects
  }) {
    const encryptedClient = encryptedSavedObjects.getClient();
    const monitors = [];
    const finder = await encryptedClient.createPointInTimeFinderDecryptedAsInternalUser({
      type: _synthetics_monitor.syntheticsMonitorType,
      perPage: 1000,
      namespaces: [spaceId]
    });
    for await (const response of finder.find()) {
      response.saved_objects.forEach(monitor => {
        monitors.push(monitor);
      });
    }

    // no need to wait here
    finder.close();
    return monitors;
  }
  mixParamsWithMonitors(spaceId, monitors, paramsBySpace) {
    const heartbeatConfigs = [];
    for (const monitor of monitors) {
      const attributes = monitor.attributes;
      const {
        str: paramsString
      } = (0, _format_configs.mixParamsWithGlobalParams)(paramsBySpace[spaceId], attributes);
      heartbeatConfigs.push((0, _format_configs.formatHeartbeatRequest)({
        monitor: (0, _utils.normalizeSecrets)(monitor).attributes,
        configId: monitor.id
      }, paramsString));
    }
    return heartbeatConfigs;
  }
}
exports.SyntheticsMonitorClient = SyntheticsMonitorClient;