"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.deleteSyntheticsMonitorRoute = exports.deleteMonitor = void 0;
var _configSchema = require("@kbn/config-schema");
var _server = require("@kbn/core/server");
var _common = require("@kbn/spaces-plugin/common");
var _synthetics_private_location = require("../../synthetics_service/private_location/synthetics_private_location");
var _runtime_types = require("../../../common/runtime_types");
var _constants = require("../../../common/constants");
var _synthetics_monitor = require("../../legacy_uptime/lib/saved_objects/synthetics_monitor");
var _service_errors = require("../synthetics_service/service_errors");
var _monitor_upgrade_sender = require("../telemetry/monitor_upgrade_sender");
var _secrets = require("../../synthetics_service/utils/secrets");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const deleteSyntheticsMonitorRoute = () => ({
  method: 'DELETE',
  path: _constants.API_URLS.SYNTHETICS_MONITORS + '/{monitorId}',
  validate: {
    params: _configSchema.schema.object({
      monitorId: _configSchema.schema.string({
        minLength: 1,
        maxLength: 1024
      })
    })
  },
  writeAccess: true,
  handler: async routeContext => {
    const {
      request,
      response
    } = routeContext;
    const {
      monitorId
    } = request.params;
    try {
      const errors = await deleteMonitor({
        routeContext,
        monitorId
      });
      if (errors && errors.length > 0) {
        return response.ok({
          body: {
            message: 'error pushing monitor to the service',
            attributes: {
              errors
            }
          }
        });
      }
      return monitorId;
    } catch (getErr) {
      if (_server.SavedObjectsErrorHelpers.isNotFoundError(getErr)) {
        return (0, _service_errors.getMonitorNotFoundResponse)(response, monitorId);
      }
      throw getErr;
    }
  }
});
exports.deleteSyntheticsMonitorRoute = deleteSyntheticsMonitorRoute;
const deleteMonitor = async ({
  routeContext,
  monitorId
}) => {
  const {
    savedObjectsClient,
    server,
    syntheticsMonitorClient,
    request
  } = routeContext;
  const {
    logger,
    telemetry,
    stackVersion
  } = server;
  const {
    monitor,
    monitorWithSecret
  } = await getMonitorToDelete(monitorId, savedObjectsClient, server);
  const {
    locations
  } = monitor.attributes;
  const hasPrivateLocation = locations.filter(loc => !loc.isServiceManaged);
  if (hasPrivateLocation.length > 0) {
    await syntheticsMonitorClient.privateLocationAPI.checkPermissions(request, (0, _synthetics_private_location.deletePermissionError)(monitor.attributes.name));
  }
  try {
    var _server$spaces$spaces, _server$spaces;
    const spaceId = (_server$spaces$spaces = (_server$spaces = server.spaces) === null || _server$spaces === void 0 ? void 0 : _server$spaces.spacesService.getSpaceId(request)) !== null && _server$spaces$spaces !== void 0 ? _server$spaces$spaces : _common.DEFAULT_SPACE_ID;
    const deleteSyncPromise = syntheticsMonitorClient.deleteMonitors([{
      ...monitor.attributes,
      id: monitor.attributes[_runtime_types.ConfigKey.MONITOR_QUERY_ID]
    }
    /* Type cast encrypted saved objects to decrypted saved objects for delete flow only.
     * Deletion does not require all monitor fields */], request, savedObjectsClient, spaceId);
    const deletePromise = savedObjectsClient.delete(_synthetics_monitor.syntheticsMonitorType, monitorId);
    const [errors] = await Promise.all([deleteSyncPromise, deletePromise]).catch(e => {
      server.logger.error(e);
      throw e;
    });
    (0, _monitor_upgrade_sender.sendTelemetryEvents)(logger, telemetry, (0, _monitor_upgrade_sender.formatTelemetryDeleteEvent)(monitor, stackVersion, new Date().toISOString(), Boolean(monitor.attributes[_runtime_types.ConfigKey.SOURCE_INLINE]), errors));
    return errors;
  } catch (e) {
    server.logger.error(`Unable to delete Synthetics monitor ${monitor.attributes[_runtime_types.ConfigKey.NAME]}`);
    if (monitorWithSecret) {
      await restoreDeletedMonitor({
        monitorId,
        normalizedMonitor: (0, _secrets.formatSecrets)({
          ...monitorWithSecret.attributes
        }),
        savedObjectsClient
      });
    }
    throw e;
  }
};
exports.deleteMonitor = deleteMonitor;
const getMonitorToDelete = async (monitorId, soClient, server) => {
  const encryptedSOClient = server.encryptedSavedObjects.getClient();
  try {
    const monitor = await encryptedSOClient.getDecryptedAsInternalUser(_synthetics_monitor.syntheticsMonitorType, monitorId);
    return {
      monitor: (0, _secrets.normalizeSecrets)(monitor),
      monitorWithSecret: (0, _secrets.normalizeSecrets)(monitor)
    };
  } catch (e) {
    server.logger.error(`Failed to decrypt monitor to delete ${monitorId}${e}`);
    (0, _monitor_upgrade_sender.sendErrorTelemetryEvents)(server.logger, server.telemetry, {
      reason: `Failed to decrypt monitor to delete ${monitorId}`,
      message: e === null || e === void 0 ? void 0 : e.message,
      type: 'deletionError',
      code: e === null || e === void 0 ? void 0 : e.code,
      status: e.status,
      stackVersion: server.stackVersion
    });
  }
  const monitor = await soClient.get(_synthetics_monitor.syntheticsMonitorType, monitorId);
  return {
    monitor,
    withSecrets: false
  };
};
const restoreDeletedMonitor = async ({
  monitorId,
  savedObjectsClient,
  normalizedMonitor
}) => {
  try {
    await savedObjectsClient.get(_synthetics_monitor.syntheticsMonitorType, monitorId);
  } catch (e) {
    if (_server.SavedObjectsErrorHelpers.isNotFoundError(e)) {
      await savedObjectsClient.create(_synthetics_monitor.syntheticsMonitorType, normalizedMonitor, {
        id: monitorId,
        overwrite: true
      });
    }
  }
};