"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.StatusRuleExecutor = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _moment = _interopRequireDefault(require("moment"));
var _constants = require("../../../common/constants");
var _get_all_locations = require("../../synthetics_service/get_all_locations");
var _get_all_monitors = require("../../saved_objects/synthetics_monitor/get_all_monitors");
var _query_monitor_status = require("../../queries/query_monitor_status");
var _lib = require("../../legacy_uptime/lib/lib");
var _runtime_types = require("../../../common/runtime_types");
var _saved_objects = require("../../../common/types/saved_objects");
var _monitor_management = require("../../../common/constants/monitor_management");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class StatusRuleExecutor {
  constructor(previousStartedAt, p, soClient, scopedClient, server, syntheticsMonitorClient) {
    (0, _defineProperty2.default)(this, "previousStartedAt", void 0);
    (0, _defineProperty2.default)(this, "params", void 0);
    (0, _defineProperty2.default)(this, "esClient", void 0);
    (0, _defineProperty2.default)(this, "soClient", void 0);
    (0, _defineProperty2.default)(this, "server", void 0);
    (0, _defineProperty2.default)(this, "syntheticsMonitorClient", void 0);
    (0, _defineProperty2.default)(this, "monitors", []);
    (0, _defineProperty2.default)(this, "locationIdNameMap", {});
    this.previousStartedAt = previousStartedAt;
    this.params = p;
    this.soClient = soClient;
    this.esClient = new _lib.UptimeEsClient(this.soClient, scopedClient, {
      heartbeatIndices: _constants.SYNTHETICS_INDEX_PATTERN
    });
    this.server = server;
    this.syntheticsMonitorClient = syntheticsMonitorClient;
  }
  async getAllLocationNames() {
    const {
      publicLocations,
      privateLocations
    } = await (0, _get_all_locations.getAllLocations)({
      server: this.server,
      syntheticsMonitorClient: this.syntheticsMonitorClient,
      savedObjectsClient: this.soClient
    });
    publicLocations.forEach(loc => {
      this.locationIdNameMap[loc.label] = loc.id;
    });
    privateLocations.forEach(loc => {
      this.locationIdNameMap[loc.label] = loc.id;
    });
  }
  getLocationId(name) {
    return this.locationIdNameMap[name];
  }
  async getMonitors() {
    await this.getAllLocationNames();
    this.monitors = await (0, _get_all_monitors.getAllMonitors)({
      soClient: this.soClient,
      filter: `${_saved_objects.monitorAttributes}.${_monitor_management.AlertConfigKey.STATUS_ENABLED}: true`
    });
    const {
      allIds,
      enabledMonitorQueryIds,
      listOfLocations,
      monitorLocationMap,
      projectMonitorsCount,
      monitorQueryIdToConfigIdMap
    } = await (0, _get_all_monitors.processMonitors)(this.monitors, this.server, this.soClient, this.syntheticsMonitorClient);
    return {
      enabledMonitorQueryIds,
      listOfLocations,
      allIds,
      monitorLocationMap,
      projectMonitorsCount,
      monitorQueryIdToConfigIdMap
    };
  }
  async getDownChecks(prevDownConfigs = {}) {
    const {
      listOfLocations,
      enabledMonitorQueryIds,
      allIds,
      monitorLocationMap,
      projectMonitorsCount,
      monitorQueryIdToConfigIdMap
    } = await this.getMonitors();
    const from = this.previousStartedAt ? (0, _moment.default)(this.previousStartedAt).subtract(1, 'minute').toISOString() : 'now-2m';
    if (enabledMonitorQueryIds.length > 0) {
      const currentStatus = await (0, _query_monitor_status.queryMonitorStatus)(this.esClient, listOfLocations, {
        to: 'now',
        from
      }, enabledMonitorQueryIds, monitorLocationMap, monitorQueryIdToConfigIdMap);
      const downConfigs = currentStatus.downConfigs;
      const upConfigs = currentStatus.upConfigs;
      Object.keys(prevDownConfigs).forEach(locId => {
        if (!downConfigs[locId] && !upConfigs[locId]) {
          downConfigs[locId] = prevDownConfigs[locId];
        }
      });
      const staleDownConfigs = this.markDeletedConfigs(downConfigs);
      return {
        ...currentStatus,
        staleDownConfigs,
        projectMonitorsCount,
        allMonitorsCount: allIds.length,
        disabledMonitorsCount: allIds.length - enabledMonitorQueryIds.length,
        allIds
      };
    }
    const staleDownConfigs = this.markDeletedConfigs(prevDownConfigs);
    return {
      downConfigs: {
        ...prevDownConfigs
      },
      upConfigs: {},
      pendingConfigs: {},
      staleDownConfigs,
      down: 0,
      up: 0,
      pending: 0,
      enabledMonitorQueryIds,
      allMonitorsCount: allIds.length,
      disabledMonitorsCount: allIds.length,
      projectMonitorsCount,
      allIds
    };
  }
  markDeletedConfigs(downConfigs) {
    const monitors = this.monitors;
    const staleDownConfigs = {};
    Object.keys(downConfigs).forEach(locPlusId => {
      const downConfig = downConfigs[locPlusId];
      const monitor = monitors.find(m => {
        return m.id === downConfig.configId || m.attributes[_runtime_types.ConfigKey.MONITOR_QUERY_ID] === downConfig.monitorQueryId;
      });
      if (!monitor) {
        staleDownConfigs[locPlusId] = {
          ...downConfig,
          isDeleted: true
        };
        delete downConfigs[locPlusId];
      } else {
        const {
          locations
        } = monitor.attributes;
        const isLocationRemoved = !locations.some(l => l.id === this.getLocationId(downConfig.location));
        if (isLocationRemoved) {
          staleDownConfigs[locPlusId] = {
            ...downConfig,
            isLocationRemoved: true
          };
          delete downConfigs[locPlusId];
        }
      }
    });
    return staleDownConfigs;
  }
}
exports.StatusRuleExecutor = StatusRuleExecutor;