"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.isHighlightedItem = exports.getSidebarItems = exports.getSeriesAndDomain = exports.getLegendItems = exports.getConnectingTime = exports.formatTooltipHeading = exports.extractItems = exports.colourPalette = void 0;
var _eui = require("@elastic/eui");
var _moment = _interopRequireDefault(require("moment"));
var _types = require("./types");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const extractItems = data => {
  // NOTE: This happens client side as the "payload" property is mapped
  // in such a way it can't be queried (or sorted on) via ES.
  return data.sort((a, b) => {
    return a.requestSentTime - b.requestSentTime;
  });
};
exports.extractItems = extractItems;
const formatValueForDisplay = (value, points = 3) => {
  return Number(value).toFixed(points);
};
const getColourForMimeType = mimeType => {
  const key = mimeType && _types.MimeTypesMap[mimeType] ? _types.MimeTypesMap[mimeType] : _types.MimeType.Other;
  return colourPalette[key];
};
const getFriendlyTooltipValue = ({
  value,
  timing,
  mimeType
}) => {
  let label = _types.FriendlyTimingLabels[timing];
  if (timing === _types.Timings.Receive && mimeType) {
    const formattedMimeType = _types.MimeTypesMap[mimeType];
    label += ` (${_types.FriendlyMimetypeLabels[formattedMimeType] || mimeType})`;
  }
  return `${label}: ${formatValueForDisplay(value)}ms`;
};
const isHighlightedItem = (item, query, activeFilters = []) => {
  var _item$url;
  if (!query && (activeFilters === null || activeFilters === void 0 ? void 0 : activeFilters.length) === 0) {
    return true;
  }
  const matchQuery = query ? (_item$url = item.url) === null || _item$url === void 0 ? void 0 : _item$url.includes(query) : true;
  const matchFilters = activeFilters.length > 0 ? activeFilters.includes(_types.MimeTypesMap[item.mimeType]) : true;
  return !!(matchQuery && matchFilters);
};
exports.isHighlightedItem = isHighlightedItem;
const getFriendlyMetadataValue = ({
  value,
  postFix
}) => {
  // value === -1 indicates timing data cannot be extracted
  if (value === undefined || value === -1) {
    return undefined;
  }
  let formattedValue = formatValueForDisplay(value);
  if (postFix) {
    formattedValue = `${formattedValue} ${postFix}`;
  }
  return formattedValue;
};
const getConnectingTime = (connect, ssl) => {
  if (ssl && connect && ssl > 0) {
    return connect - ssl;
  } else {
    return connect;
  }
};
exports.getConnectingTime = getConnectingTime;
const getSeriesAndDomain = (items, onlyHighlighted = false, query, activeFilters) => {
  const getValueForOffset = item => {
    return item.requestSentTime;
  };
  // The earliest point in time a request is sent or started. This will become our notion of "0".
  let zeroOffset = Infinity;
  items.forEach(i => zeroOffset = Math.min(zeroOffset, getValueForOffset(i)));
  const getValue = (timings, timing) => {
    if (!timings) return;

    // SSL is a part of the connect timing
    if (timing === _types.Timings.Connect) {
      return getConnectingTime(timings.connect, timings.ssl);
    }
    return timings[timing];
  };
  const series = [];
  const metadata = [];
  let totalHighlightedRequests = 0;
  items.forEach((item, index) => {
    const mimeTypeColour = getColourForMimeType(item.mimeType);
    const offsetValue = getValueForOffset(item);
    let currentOffset = offsetValue - zeroOffset;
    metadata.push(formatMetadata({
      item,
      index,
      requestStart: currentOffset
    }));
    const isHighlighted = isHighlightedItem(item, query, activeFilters);
    if (isHighlighted) {
      totalHighlightedRequests++;
    }
    if (!item.timings) {
      series.push({
        x: index,
        y0: 0,
        y: 0,
        config: {
          isHighlighted,
          showTooltip: false
        }
      });
      return;
    }
    let timingValueFound = false;
    _types.TIMING_ORDER.forEach(timing => {
      const value = getValue(item.timings, timing);
      if (value && value >= 0) {
        timingValueFound = true;
        const colour = timing === _types.Timings.Receive ? mimeTypeColour : colourPalette[timing];
        const y = currentOffset + value;
        series.push({
          x: index,
          y0: currentOffset,
          y,
          config: {
            id: index,
            colour,
            isHighlighted,
            showTooltip: true,
            tooltipProps: {
              value: getFriendlyTooltipValue({
                value: y - currentOffset,
                timing,
                mimeType: item.mimeType
              }),
              colour
            }
          }
        });
        currentOffset = y;
      }
    });

    /* if no specific timing values are found, use the total time
     * if total time is not available use 0, set showTooltip to false,
     * and omit tooltip props */
    if (!timingValueFound) {
      const total = item.timings.total;
      const hasTotal = total !== -1;
      series.push({
        x: index,
        y0: hasTotal ? currentOffset : 0,
        y: hasTotal ? currentOffset + item.timings.total : 0,
        config: {
          isHighlighted,
          colour: hasTotal ? mimeTypeColour : '',
          showTooltip: hasTotal,
          tooltipProps: hasTotal ? {
            value: getFriendlyTooltipValue({
              value: total,
              timing: _types.Timings.Receive,
              mimeType: item.mimeType
            }),
            colour: mimeTypeColour
          } : undefined
        }
      });
    }
  });
  const yValues = series.map(serie => serie.y);
  const domain = {
    min: 0,
    max: Math.max(...yValues)
  };
  let filteredSeries = series;
  if (onlyHighlighted) {
    filteredSeries = series.filter(item => item.config.isHighlighted);
  }
  return {
    series: filteredSeries,
    domain,
    metadata,
    totalHighlightedRequests
  };
};
exports.getSeriesAndDomain = getSeriesAndDomain;
const formatHeaders = headers => {
  if (typeof headers === 'undefined') {
    return undefined;
  }
  return Object.keys(headers).map(key => ({
    name: key,
    value: `${headers[key]}`
  }));
};
const formatMetadata = ({
  item,
  index,
  requestStart
}) => {
  const {
    certificates,
    ip,
    mimeType,
    requestHeaders,
    responseHeaders,
    url,
    resourceSize,
    transferSize,
    status
  } = item;
  const {
    dns,
    connect,
    ssl,
    wait,
    receive,
    total
  } = item.timings || {};
  const contentDownloaded = receive && receive > 0 ? receive : total;
  return {
    x: index,
    url,
    requestHeaders: formatHeaders(requestHeaders),
    responseHeaders: formatHeaders(responseHeaders),
    certificates: certificates ? [{
      name: _types.FriendlyFlyoutLabels[_types.Metadata.CertificateIssuer],
      value: certificates.issuer
    }, {
      name: _types.FriendlyFlyoutLabels[_types.Metadata.CertificateIssueDate],
      value: certificates.validFrom ? (0, _moment.default)(certificates.validFrom).format('L LT') : undefined
    }, {
      name: _types.FriendlyFlyoutLabels[_types.Metadata.CertificateExpiryDate],
      value: certificates.validTo ? (0, _moment.default)(certificates.validTo).format('L LT') : undefined
    }, {
      name: _types.FriendlyFlyoutLabels[_types.Metadata.CertificateSubject],
      value: certificates.subjectName
    }] : undefined,
    details: [{
      name: _types.FriendlyFlyoutLabels[_types.Metadata.Status],
      value: status ? `${status}` : undefined
    }, {
      name: _types.FriendlyFlyoutLabels[_types.Metadata.MimeType],
      value: mimeType
    }, {
      name: _types.FriendlyFlyoutLabels[_types.Metadata.RequestStart],
      value: getFriendlyMetadataValue({
        value: requestStart,
        postFix: 'ms'
      })
    }, {
      name: _types.FriendlyTimingLabels[_types.Timings.Dns],
      value: getFriendlyMetadataValue({
        value: dns,
        postFix: 'ms'
      })
    }, {
      name: _types.FriendlyTimingLabels[_types.Timings.Connect],
      value: getFriendlyMetadataValue({
        value: getConnectingTime(connect, ssl),
        postFix: 'ms'
      })
    }, {
      name: _types.FriendlyTimingLabels[_types.Timings.Ssl],
      value: getFriendlyMetadataValue({
        value: ssl,
        postFix: 'ms'
      })
    }, {
      name: _types.FriendlyTimingLabels[_types.Timings.Wait],
      value: getFriendlyMetadataValue({
        value: wait,
        postFix: 'ms'
      })
    }, {
      name: _types.FriendlyTimingLabels[_types.Timings.Receive],
      value: getFriendlyMetadataValue({
        value: contentDownloaded,
        postFix: 'ms'
      })
    }, {
      name: _types.FriendlyFlyoutLabels[_types.Metadata.ResourceSize],
      value: getFriendlyMetadataValue({
        value: resourceSize ? resourceSize / 1000 : undefined,
        postFix: 'KB'
      })
    }, {
      name: _types.FriendlyFlyoutLabels[_types.Metadata.TransferSize],
      value: getFriendlyMetadataValue({
        value: transferSize ? transferSize / 1000 : undefined,
        postFix: 'KB'
      })
    }, {
      name: _types.FriendlyFlyoutLabels[_types.Metadata.IP],
      value: ip
    }]
  };
};
const getSidebarItems = (items, onlyHighlighted, query, activeFilters) => {
  const sideBarItems = items.map((item, index) => {
    const isHighlighted = isHighlightedItem(item, query, activeFilters);
    const offsetIndex = index + 1;
    const {
      url,
      status,
      method
    } = item;
    return {
      url,
      status,
      method,
      isHighlighted,
      offsetIndex,
      index
    };
  });
  if (onlyHighlighted) {
    return sideBarItems.filter(item => item.isHighlighted);
  }
  return sideBarItems;
};
exports.getSidebarItems = getSidebarItems;
const getLegendItems = () => {
  let timingItems = [];
  Object.values(_types.Timings).forEach(timing => {
    // The "receive" timing is mapped to a mime type colour, so we don't need to show this in the legend
    if (timing === _types.Timings.Receive) {
      return;
    }
    timingItems = [...timingItems, {
      name: _types.FriendlyTimingLabels[timing],
      colour: TIMING_PALETTE[timing]
    }];
  });
  let mimeTypeItems = [];
  Object.values(_types.MimeType).forEach(mimeType => {
    mimeTypeItems = [...mimeTypeItems, {
      name: _types.FriendlyMimetypeLabels[mimeType],
      colour: MIME_TYPE_PALETTE[mimeType]
    }];
  });
  return [...timingItems, ...mimeTypeItems];
};

// Timing colour palette
exports.getLegendItems = getLegendItems;
const SAFE_PALETTE = (0, _eui.euiPaletteColorBlind)({
  rotations: 2
});
const buildTimingPalette = () => {
  const palette = Object.values(_types.Timings).reduce((acc, value) => {
    switch (value) {
      case _types.Timings.Blocked:
        acc[value] = SAFE_PALETTE[16];
        break;
      case _types.Timings.Dns:
        acc[value] = SAFE_PALETTE[0];
        break;
      case _types.Timings.Connect:
        acc[value] = SAFE_PALETTE[7];
        break;
      case _types.Timings.Ssl:
        acc[value] = SAFE_PALETTE[17];
        break;
      case _types.Timings.Send:
        acc[value] = SAFE_PALETTE[2];
        break;
      case _types.Timings.Wait:
        acc[value] = SAFE_PALETTE[11];
        break;
      case _types.Timings.Receive:
        acc[value] = SAFE_PALETTE[0];
        break;
    }
    return acc;
  }, {});
  return palette;
};
const TIMING_PALETTE = buildTimingPalette();

// MimeType colour palette

const buildMimeTypePalette = () => {
  const palette = Object.values(_types.MimeType).reduce((acc, value) => {
    switch (value) {
      case _types.MimeType.Html:
        acc[value] = SAFE_PALETTE[19];
        break;
      case _types.MimeType.Script:
        acc[value] = SAFE_PALETTE[3];
        break;
      case _types.MimeType.Stylesheet:
        acc[value] = SAFE_PALETTE[4];
        break;
      case _types.MimeType.Media:
        acc[value] = SAFE_PALETTE[5];
        break;
      case _types.MimeType.Font:
        acc[value] = SAFE_PALETTE[8];
        break;
      case _types.MimeType.XHR:
      case _types.MimeType.Other:
        acc[value] = SAFE_PALETTE[9];
        break;
    }
    return acc;
  }, {});
  return palette;
};
const MIME_TYPE_PALETTE = buildMimeTypePalette();
const colourPalette = {
  ...TIMING_PALETTE,
  ...MIME_TYPE_PALETTE
};
exports.colourPalette = colourPalette;
const formatTooltipHeading = (index, fullText) => isNaN(index) ? fullText : `${index}. ${fullText}`;
exports.formatTooltipHeading = formatTooltipHeading;