"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getConnectorType = void 0;
var _types = require("@kbn/actions-plugin/common/types");
var _mustache_renderer = require("@kbn/actions-plugin/server/lib/mustache_renderer");
var _i18n = require("@kbn/i18n");
var _configSchema = require("@kbn/config-schema");
var _schema = require("../../../common/slack_api/schema");
var _constants = require("../../../common/slack_api/constants");
var _translations = require("./translations");
var _api = require("./api");
var _service = require("./service");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const supportedSubActions = ['getChannels', 'postMessage'];
const getConnectorType = () => {
  return {
    id: _constants.SLACK_API_CONNECTOR_ID,
    minimumLicenseRequired: 'gold',
    name: _translations.SLACK_CONNECTOR_NAME,
    supportedFeatureIds: [_types.AlertingConnectorFeatureId, _types.SecurityConnectorFeatureId],
    validate: {
      config: {
        schema: _configSchema.schema.object({}, {
          defaultValue: {}
        })
      },
      secrets: {
        schema: _schema.SlackApiSecretsSchema,
        customValidator: validateSlackUrl
      },
      params: {
        schema: _schema.SlackApiParamsSchema
      }
    },
    renderParameterTemplates,
    executor: async execOptions => await slackApiExecutor(execOptions)
  };
};
exports.getConnectorType = getConnectorType;
const validateSlackUrl = (secretsObject, validatorServices) => {
  const {
    configurationUtilities
  } = validatorServices;
  try {
    configurationUtilities.ensureUriAllowed(_constants.SLACK_URL);
  } catch (allowedListError) {
    throw new Error(_i18n.i18n.translate('xpack.stackConnectors.slack_api.configurationError', {
      defaultMessage: 'error configuring slack action: {message}',
      values: {
        message: allowedListError.message
      }
    }));
  }
};
const renderParameterTemplates = (params, variables) => {
  if (params.subAction === 'postMessage') return {
    subAction: params.subAction,
    subActionParams: {
      ...params.subActionParams,
      text: (0, _mustache_renderer.renderMustacheString)(params.subActionParams.text, variables, 'slack')
    }
  };
  return params;
};
const slackApiExecutor = async ({
  actionId,
  params,
  secrets,
  configurationUtilities,
  logger
}) => {
  const subAction = params.subAction;
  if (!_api.api[subAction]) {
    const errorMessage = `[Action][ExternalService] -> [Slack API] Unsupported subAction type ${subAction}.`;
    logger.error(errorMessage);
    throw new Error(errorMessage);
  }
  if (!supportedSubActions.includes(subAction)) {
    const errorMessage = `[Action][ExternalService] -> [Slack API] subAction ${subAction} not implemented.`;
    logger.error(errorMessage);
    throw new Error(errorMessage);
  }
  const externalService = (0, _service.createExternalService)({
    secrets
  }, logger, configurationUtilities);
  if (subAction === 'getChannels') {
    return await _api.api.getChannels({
      externalService
    });
  }
  if (subAction === 'postMessage') {
    return await _api.api.postMessage({
      externalService,
      params: params.subActionParams
    });
  }
  return {
    status: 'ok',
    data: {},
    actionId
  };
};