"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.EsQueryRuleParamsSchema = exports.ES_QUERY_MAX_HITS_PER_EXECUTION = void 0;
var _i18n = require("@kbn/i18n");
var _configSchema = require("@kbn/config-schema");
var _server = require("@kbn/triggers-actions-ui-plugin/server");
var _common = require("../../../common");
var _comparator = require("../lib/comparator");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const ES_QUERY_MAX_HITS_PER_EXECUTION = 10000;

// rule type parameters
exports.ES_QUERY_MAX_HITS_PER_EXECUTION = ES_QUERY_MAX_HITS_PER_EXECUTION;
const EsQueryRuleParamsSchemaProperties = {
  size: _configSchema.schema.number({
    min: 0,
    max: ES_QUERY_MAX_HITS_PER_EXECUTION
  }),
  timeWindowSize: _configSchema.schema.number({
    min: 1
  }),
  excludeHitsFromPreviousRun: _configSchema.schema.boolean({
    defaultValue: true
  }),
  timeWindowUnit: _configSchema.schema.string({
    validate: _server.validateTimeWindowUnits
  }),
  threshold: _configSchema.schema.arrayOf(_configSchema.schema.number(), {
    minSize: 1,
    maxSize: 2
  }),
  thresholdComparator: (0, _comparator.getComparatorSchemaType)(validateComparator),
  // aggregation type
  aggType: _configSchema.schema.string({
    validate: _server.validateAggType,
    defaultValue: 'count'
  }),
  // aggregation field
  aggField: _configSchema.schema.maybe(_configSchema.schema.string({
    minLength: 1
  })),
  // how to group
  groupBy: _configSchema.schema.string({
    validate: _server.validateGroupBy,
    defaultValue: 'all'
  }),
  // field to group on (for groupBy: top)
  termField: _configSchema.schema.maybe(_configSchema.schema.string({
    minLength: 1
  })),
  // limit on number of groups returned
  termSize: _configSchema.schema.maybe(_configSchema.schema.number({
    min: 1
  })),
  searchType: _configSchema.schema.oneOf([_configSchema.schema.literal('searchSource'), _configSchema.schema.literal('esQuery')], {
    defaultValue: 'esQuery'
  }),
  timeField: _configSchema.schema.conditional(_configSchema.schema.siblingRef('searchType'), _configSchema.schema.literal('esQuery'), _configSchema.schema.string({
    minLength: 1
  }), _configSchema.schema.maybe(_configSchema.schema.string({
    minLength: 1
  }))),
  // searchSource rule param only
  searchConfiguration: _configSchema.schema.conditional(_configSchema.schema.siblingRef('searchType'), _configSchema.schema.literal('searchSource'), _configSchema.schema.object({}, {
    unknowns: 'allow'
  }), _configSchema.schema.never()),
  // esQuery rule params only
  esQuery: _configSchema.schema.conditional(_configSchema.schema.siblingRef('searchType'), _configSchema.schema.literal('esQuery'), _configSchema.schema.string({
    minLength: 1
  }), _configSchema.schema.never()),
  index: _configSchema.schema.conditional(_configSchema.schema.siblingRef('searchType'), _configSchema.schema.literal('esQuery'), _configSchema.schema.arrayOf(_configSchema.schema.string({
    minLength: 1
  }), {
    minSize: 1
  }), _configSchema.schema.never())
};
const EsQueryRuleParamsSchema = _configSchema.schema.object(EsQueryRuleParamsSchemaProperties, {
  validate: validateParams
});
exports.EsQueryRuleParamsSchema = EsQueryRuleParamsSchema;
const betweenComparators = new Set(['between', 'notBetween']);

// using direct type not allowed, circular reference, so body is typed to any
function validateParams(anyParams) {
  const {
    esQuery,
    thresholdComparator,
    threshold,
    searchType,
    aggType,
    aggField,
    groupBy,
    termField,
    termSize
  } = anyParams;
  if (betweenComparators.has(thresholdComparator) && threshold.length === 1) {
    return _i18n.i18n.translate('xpack.stackAlerts.esQuery.invalidThreshold2ErrorMessage', {
      defaultMessage: '[threshold]: must have two elements for the "{thresholdComparator}" comparator',
      values: {
        thresholdComparator
      }
    });
  }
  if (aggType !== 'count' && !aggField) {
    return _i18n.i18n.translate('xpack.stackAlerts.esQuery.aggTypeRequiredErrorMessage', {
      defaultMessage: '[aggField]: must have a value when [aggType] is "{aggType}"',
      values: {
        aggType
      }
    });
  }

  // check grouping
  if (groupBy === 'top') {
    if (termField == null) {
      return _i18n.i18n.translate('xpack.stackAlerts.esQuery.termFieldRequiredErrorMessage', {
        defaultMessage: '[termField]: termField required when [groupBy] is top'
      });
    }
    if (termSize == null) {
      return _i18n.i18n.translate('xpack.stackAlerts.esQuery.termSizeRequiredErrorMessage', {
        defaultMessage: '[termSize]: termSize required when [groupBy] is top'
      });
    }
    if (termSize > _server.MAX_GROUPS) {
      return _i18n.i18n.translate('xpack.stackAlerts.esQuery.invalidTermSizeMaximumErrorMessage', {
        defaultMessage: '[termSize]: must be less than or equal to {maxGroups}',
        values: {
          maxGroups: _server.MAX_GROUPS
        }
      });
    }
  }
  if (searchType === 'searchSource') {
    return;
  }
  try {
    const parsedQuery = JSON.parse(esQuery);
    if (parsedQuery && !parsedQuery.query) {
      return _i18n.i18n.translate('xpack.stackAlerts.esQuery.missingEsQueryErrorMessage', {
        defaultMessage: '[esQuery]: must contain "query"'
      });
    }
  } catch (err) {
    return _i18n.i18n.translate('xpack.stackAlerts.esQuery.invalidEsQueryErrorMessage', {
      defaultMessage: '[esQuery]: must be valid JSON'
    });
  }
}
function validateComparator(comparator) {
  if (_common.ComparatorFnNames.has(comparator)) return;
  return _i18n.i18n.translate('xpack.stackAlerts.esQuery.invalidComparatorErrorMessage', {
    defaultMessage: 'invalid thresholdComparator specified: {comparator}',
    values: {
      comparator
    }
  });
}