"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.fetchSearchSourceQuery = fetchSearchSourceQuery;
exports.getSmallerDataViewSpec = getSmallerDataViewSpec;
exports.updateSearchSource = updateSearchSource;
var _lodash = require("lodash");
var _esQuery = require("@kbn/es-query");
var _common = require("@kbn/data-plugin/common");
var _common2 = require("@kbn/triggers-actions-ui-plugin/common");
var _common3 = require("../../../../common");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function fetchSearchSourceQuery({
  ruleId,
  alertLimit,
  params,
  latestTimestamp,
  spacePrefix,
  services
}) {
  const {
    logger,
    searchSourceClient
  } = services;
  const isGroupAgg = (0, _common2.isGroupAggregation)(params.termField);
  const isCountAgg = (0, _common2.isCountAggregation)(params.aggType);
  const initialSearchSource = await searchSourceClient.create(params.searchConfiguration);
  const index = initialSearchSource.getField('index');
  const {
    searchSource,
    dateStart,
    dateEnd
  } = updateSearchSource(initialSearchSource, index, params, latestTimestamp, alertLimit);
  logger.debug(`search source query rule (${ruleId}) query: ${JSON.stringify(searchSource.getSearchRequestBody())}`);
  const searchResult = await searchSource.fetch();
  const link = await generateLink(initialSearchSource, services.share.url.locators.get('DISCOVER_APP_LOCATOR'), services.dataViews, index, dateStart, dateEnd, spacePrefix);
  return {
    link,
    numMatches: Number(searchResult.hits.total),
    searchResult,
    parsedResults: (0, _common2.parseAggregationResults)({
      isCountAgg,
      isGroupAgg,
      esResult: searchResult
    }),
    dateStart,
    dateEnd
  };
}
function updateSearchSource(searchSource, index, params, latestTimestamp, alertLimit) {
  const isGroupAgg = (0, _common2.isGroupAggregation)(params.termField);
  const timeFieldName = params.timeField || index.timeFieldName;
  if (!timeFieldName) {
    throw new Error('Invalid data view without timeFieldName.');
  }
  searchSource.setField('size', isGroupAgg ? 0 : params.size);
  const timeRange = {
    from: `now-${params.timeWindowSize}${params.timeWindowUnit}`,
    to: 'now'
  };
  const timerangeFilter = (0, _common.getTime)(index, timeRange);
  const dateStart = timerangeFilter === null || timerangeFilter === void 0 ? void 0 : timerangeFilter.query.range[timeFieldName].gte;
  const dateEnd = timerangeFilter === null || timerangeFilter === void 0 ? void 0 : timerangeFilter.query.range[timeFieldName].lte;
  const filters = [timerangeFilter];
  if (params.excludeHitsFromPreviousRun) {
    if (latestTimestamp && latestTimestamp > dateStart) {
      // add additional filter for documents with a timestamp greater then
      // the timestamp of the previous run, so that those documents are not counted twice
      const field = index.fields.find(f => f.name === timeFieldName);
      const addTimeRangeField = (0, _esQuery.buildRangeFilter)(field, {
        gt: latestTimestamp,
        format: 'strict_date_optional_time'
      }, index);
      filters.push(addTimeRangeField);
    }
  }
  const searchSourceChild = searchSource.createChild();
  searchSourceChild.setField('filter', filters);
  searchSourceChild.setField('sort', [{
    [timeFieldName]: {
      order: _common.SortDirection.desc,
      format: 'strict_date_optional_time||epoch_millis'
    }
  }]);
  searchSourceChild.setField('aggs', (0, _common2.buildAggregation)({
    aggType: params.aggType,
    aggField: params.aggField,
    termField: params.termField,
    termSize: params.termSize,
    condition: {
      resultLimit: alertLimit,
      conditionScript: (0, _common3.getComparatorScript)(params.thresholdComparator, params.threshold, _common2.BUCKET_SELECTOR_FIELD)
    },
    ...(isGroupAgg ? {
      topHitsSize: params.size
    } : {})
  }));
  return {
    searchSource: searchSourceChild,
    dateStart,
    dateEnd
  };
}
async function generateLink(searchSource, discoverLocator, dataViews, dataViewToUpdate, dateStart, dateEnd, spacePrefix) {
  const prevFilters = searchSource.getField('filter');

  // make new adhoc data view
  const newDataView = await dataViews.create({
    ...dataViewToUpdate.toSpec(false),
    version: undefined,
    id: undefined
  });
  const updatedFilters = updateFilterReferences(prevFilters, dataViewToUpdate.id, newDataView.id);
  const redirectUrlParams = {
    dataViewSpec: getSmallerDataViewSpec(newDataView),
    filters: updatedFilters,
    query: searchSource.getField('query'),
    timeRange: {
      from: dateStart,
      to: dateEnd
    },
    isAlertResults: true
  };

  // use `lzCompress` flag for making the link readable during debugging/testing
  // const redirectUrl = discoverLocator!.getRedirectUrl(redirectUrlParams, { lzCompress: false });
  const redirectUrl = discoverLocator.getRedirectUrl(redirectUrlParams);
  const [start, end] = redirectUrl.split('/app');
  return start + spacePrefix + '/app' + end;
}
function updateFilterReferences(filters, fromDataView, toDataView) {
  return (filters || []).map(filter => {
    if (filter.meta.index === fromDataView) {
      return {
        ...filter,
        meta: {
          ...filter.meta,
          index: toDataView
        }
      };
    } else {
      return filter;
    }
  });
}
function getSmallerDataViewSpec(dataView) {
  const dataViewSpec = dataView.toSpec(false);
  if (dataViewSpec.fieldAttrs) {
    // remove `count` props
    dataViewSpec.fieldAttrs = (0, _lodash.pickBy)((0, _lodash.mapValues)(dataViewSpec.fieldAttrs, fieldAttrs => (0, _lodash.omit)(fieldAttrs, 'count')), trimmedFieldAttrs => Object.keys(trimmedFieldAttrs).length > 0);
    if (Object.keys(dataViewSpec.fieldAttrs).length === 0) {
      dataViewSpec.fieldAttrs = undefined;
    }
  }
  return dataViewSpec;
}