"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.BrowserShortUrlClient = void 0;
var _url = require("url");
var _legacy_short_url_locator = require("../../../common/url_service/locators/legacy_short_url_locator");
var _short_url_redirect_locator = require("../../../common/url_service/locators/short_url_redirect_locator");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

class BrowserShortUrlClient {
  constructor(dependencies) {
    this.dependencies = dependencies;
  }
  async create({
    locator,
    params,
    slug = undefined
  }) {
    const {
      http
    } = this.dependencies;
    const data = await http.fetch('/api/short_url', {
      method: 'POST',
      body: JSON.stringify({
        locatorId: locator.id,
        slug,
        params
      })
    });
    return {
      data
    };
  }
  async createWithLocator(params) {
    const result = await this.create(params);
    const redirectLocator = this.dependencies.locators.get(_short_url_redirect_locator.SHORT_URL_REDIRECT_LOCATOR);
    const redirectParams = {
      slug: result.data.slug
    };
    return {
      ...result,
      locator: redirectLocator,
      params: redirectParams
    };
  }
  async createFromLongUrl(longUrl) {
    const parsedUrl = (0, _url.parse)(longUrl);
    if (!parsedUrl || !parsedUrl.path) {
      throw new Error(`Invalid URL: ${longUrl}`);
    }
    const path = parsedUrl.path.replace(this.dependencies.http.basePath.get(), '');
    const hash = parsedUrl.hash ? parsedUrl.hash : '';
    const relativeUrl = path + hash;
    const locator = this.dependencies.locators.get(_legacy_short_url_locator.LEGACY_SHORT_URL_LOCATOR_ID);
    if (!locator) {
      throw new Error(`Locator "${_legacy_short_url_locator.LEGACY_SHORT_URL_LOCATOR_ID}" not found`);
    }
    const result = await this.createWithLocator({
      locator,
      params: {
        url: relativeUrl
      }
    });
    const shortUrl = await result.locator.getUrl(result.params, {
      absolute: true
    });
    return {
      ...result,
      url: shortUrl
    };
  }
  async get(id) {
    const {
      http
    } = this.dependencies;
    const data = await http.fetch('/api/short_url/' + id, {
      method: 'GET'
    });
    return {
      data
    };
  }
  async resolve(slug) {
    const {
      http
    } = this.dependencies;
    const data = await http.fetch('/api/short_url/_slug/' + slug, {
      method: 'GET'
    });
    return {
      data
    };
  }
  async delete(id) {
    const {
      http
    } = this.dependencies;
    await http.fetch('/api/short_url/' + id, {
      method: 'DELETE'
    });
  }
}
exports.BrowserShortUrlClient = BrowserShortUrlClient;