"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.transformToNotifyWhen = exports.transformToFrequency = exports.transformToAlertThrottle = exports.transformToActionFrequency = exports.transformFromAlertThrottle = void 0;
var _constants = require("../../../../../common/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const transformToFrequency = throttle => {
  var _transformToNotifyWhe;
  return {
    summary: true,
    notifyWhen: (_transformToNotifyWhe = transformToNotifyWhen(throttle)) !== null && _transformToNotifyWhe !== void 0 ? _transformToNotifyWhe : 'onActiveAlert',
    throttle: transformToAlertThrottle(throttle)
  };
};
exports.transformToFrequency = transformToFrequency;
/**
 * The action level `frequency` attribute should always take precedence over the rule level `throttle`
 * Frequency's default value is `{ summary: true, throttle: null, notifyWhen: 'onActiveAlert' }`
 *
 * The transformation follows the next rules:
 * - Both rule level `throttle` and all actions have `frequency` are set: we will ignore rule level `throttle`
 * - Rule level `throttle` set and actions don't have `frequency` set: we will transform rule level `throttle` in action level `frequency`
 * - All actions have `frequency` set: do nothing
 * - Neither of them is set: we will set action level `frequency` to default value
 * - Rule level `throttle` and some of the actions have `frequency` set: we will transform rule level `throttle` and set it to actions without the frequency attribute
 * - Only some actions have `frequency` set and there is no rule level `throttle`: we will set default `frequency` to actions without frequency attribute
 */
const transformToActionFrequency = (actions, throttle) => {
  const defaultFrequency = transformToFrequency(throttle);
  return actions.map(action => {
    var _action$frequency;
    return {
      ...action,
      frequency: (_action$frequency = action.frequency) !== null && _action$frequency !== void 0 ? _action$frequency : defaultFrequency
    };
  });
};

/**
 * Given a throttle from a "security_solution" rule this will transform it into an "alerting" notifyWhen
 * on their saved object.
 * @params throttle The throttle from a "security_solution" rule
 * @returns The correct "NotifyWhen" for a Kibana alerting.
 */
exports.transformToActionFrequency = transformToActionFrequency;
const transformToNotifyWhen = throttle => {
  if (throttle == null || throttle === _constants.NOTIFICATION_THROTTLE_NO_ACTIONS) {
    return null; // Although I return null, this does not change the value of the "notifyWhen" and it keeps the current value of "notifyWhen"
  } else if (throttle === _constants.NOTIFICATION_THROTTLE_RULE) {
    return 'onActiveAlert';
  } else {
    return 'onThrottleInterval';
  }
};

/**
 * Given a throttle from a "security_solution" rule this will transform it into an "alerting" "throttle"
 * on their saved object.
 * @params throttle The throttle from a "security_solution" rule
 * @returns The "alerting" throttle
 */
exports.transformToNotifyWhen = transformToNotifyWhen;
const transformToAlertThrottle = throttle => {
  if (throttle == null || throttle === _constants.NOTIFICATION_THROTTLE_RULE || throttle === _constants.NOTIFICATION_THROTTLE_NO_ACTIONS) {
    return null;
  } else {
    return throttle;
  }
};

/**
 * Given a throttle from an "alerting" Saved Object (SO) this will transform it into a "security_solution"
 * throttle type.
 * @param throttle The throttle from an "alerting" Saved Object (SO)
 * @returns The "security_solution" throttle
 */
exports.transformToAlertThrottle = transformToAlertThrottle;
const transformFromAlertThrottle = rule => {
  var _rule$throttle;
  if (rule.notifyWhen == null) {
    return transformFromFirstActionThrottle(rule);
  } else if (rule.notifyWhen === 'onActiveAlert') {
    return _constants.NOTIFICATION_THROTTLE_RULE;
  }
  return (_rule$throttle = rule.throttle) !== null && _rule$throttle !== void 0 ? _rule$throttle : undefined;
};
exports.transformFromAlertThrottle = transformFromAlertThrottle;
function transformFromFirstActionThrottle(rule) {
  var _rule$actions$0$frequ, _rule$actions$;
  const frequency = (_rule$actions$0$frequ = (_rule$actions$ = rule.actions[0]) === null || _rule$actions$ === void 0 ? void 0 : _rule$actions$.frequency) !== null && _rule$actions$0$frequ !== void 0 ? _rule$actions$0$frequ : null;
  if (!frequency || frequency.notifyWhen !== 'onThrottleInterval' || frequency.throttle == null) return _constants.NOTIFICATION_THROTTLE_RULE;
  return frequency.throttle;
}