"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.importRuleActionConnectors = void 0;
var _stream = require("stream");
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const importRuleActionConnectors = async ({
  actionConnectors,
  actionsClient,
  actionsImporter,
  rules,
  overwrite
}) => {
  try {
    const actionConnectorRules = (0, _utils.getActionConnectorRules)(rules);
    const actionsIds = Object.keys(actionConnectorRules);
    if (!actionsIds.length) return {
      success: true,
      errors: [],
      successCount: 0,
      warnings: []
    };
    if (overwrite && !actionConnectors.length) return (0, _utils.handleActionsHaveNoConnectors)(actionsIds, actionConnectorRules);
    let actionConnectorsToImport = actionConnectors;
    if (!overwrite) {
      const newIdsToAdd = await (0, _utils.filterExistingActionConnectors)(actionsClient, actionsIds);
      const foundMissingConnectors = (0, _utils.checkIfActionsHaveMissingConnectors)(actionConnectors, newIdsToAdd, actionConnectorRules);
      if (foundMissingConnectors) return foundMissingConnectors;
      // filter out existing connectors
      actionConnectorsToImport = actionConnectors.filter(({
        id
      }) => newIdsToAdd.includes(id));
    }
    if (!actionConnectorsToImport.length) return {
      success: true,
      errors: [],
      successCount: 0,
      warnings: []
    };
    const readStream = _stream.Readable.from(actionConnectorsToImport);
    const {
      success,
      successCount,
      successResults,
      warnings,
      errors
    } = await actionsImporter.import({
      readStream,
      overwrite,
      createNewCopies: false
    });
    /*
      // When a connector is exported from one namespace and imported to another, it does not result in an error, but instead a new object is created with
      // new destination id and id will have the old  origin id, so in order to be able to use the newly generated Connectors id, this util is used to swap the old id with the
      // new destination Id
      */
    let rulesWithMigratedActions;
    if (successResults !== null && successResults !== void 0 && successResults.some(res => res.destinationId)) rulesWithMigratedActions = (0, _utils.updateRuleActionsWithMigratedResults)(rules, successResults);
    return {
      success,
      successCount,
      errors: errors ? (0, _utils.mapSOErrorToRuleError)(errors) : [],
      warnings: warnings || [],
      rulesWithMigratedActions
    };
  } catch (error) {
    return (0, _utils.returnErroredImportResult)(error);
  }
};
exports.importRuleActionConnectors = importRuleActionConnectors;