"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getRuleActionConnectorsForExport = void 0;
var _utils = require("@kbn/utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const defaultActionConnectorDetails = {
  exported_action_connector_count: 0,
  missing_action_connection_count: 0,
  missing_action_connections: [],
  excluded_action_connection_count: 0,
  excluded_action_connections: []
};
const mapExportedActionConnectorsDetailsToDefault = exportDetails => {
  return {
    exported_action_connector_count: exportDetails.exportedCount,
    missing_action_connection_count: exportDetails.missingRefCount,
    missing_action_connections: exportDetails.missingReferences,
    excluded_action_connection_count: exportDetails.excludedObjectsCount,
    excluded_action_connections: exportDetails.excludedObjects
  };
};
const filterOutPredefinedActionConnectorsIds = async (actionsClient, actionsIdsToExport) => {
  const allActions = await actionsClient.getAll();
  const predefinedActionsIds = allActions.filter(({
    isPreconfigured
  }) => isPreconfigured).map(({
    id
  }) => id);
  if (predefinedActionsIds.length) return actionsIdsToExport.filter(id => !predefinedActionsIds.includes(id));
  return actionsIdsToExport;
};

// This function is used to get, and return the exported actions' connectors'
// using the ISavedObjectsExporter and it filters out any Preconfigured
// Connectors as they shouldn't be exported, imported or changed
// by the user, that's why the function also accepts the actionsClient
// to getAll actions connectors

const getRuleActionConnectorsForExport = async (rules, actionsExporter, request, actionsClient) => {
  const exportedActionConnectors = {
    actionConnectors: '',
    actionConnectorDetails: defaultActionConnectorDetails
  };
  let actionsIds = [...new Set(rules.flatMap(rule => rule.actions.map(({
    id
  }) => id)))];
  if (!actionsIds.length) return exportedActionConnectors;

  // handle preconfigured connectors
  actionsIds = await filterOutPredefinedActionConnectorsIds(actionsClient, actionsIds);
  if (!actionsIds.length) return exportedActionConnectors;
  const getActionsByObjectsParam = actionsIds.map(id => ({
    type: 'action',
    id
  }));
  const actionDetails = await actionsExporter.exportByObjects({
    objects: getActionsByObjectsParam,
    request
  });
  if (!actionDetails) {
    exportedActionConnectors.actionConnectorDetails = {
      exported_action_connector_count: 0,
      missing_action_connection_count: actionsIds.length,
      missing_action_connections: [],
      // TODO: check how to generate SO
      excluded_action_connection_count: 0,
      excluded_action_connections: []
    };
    return exportedActionConnectors;
  }
  const actionsConnectorsToExport = await (0, _utils.createPromiseFromStreams)([actionDetails, (0, _utils.createMapStream)(obj => {
    if ('exportedCount' in obj) exportedActionConnectors.actionConnectorDetails = mapExportedActionConnectorsDetailsToDefault(obj);else return JSON.stringify(obj);
  }), (0, _utils.createConcatStream)([])]);
  exportedActionConnectors.actionConnectors = actionsConnectorsToExport.join('\n');
  return exportedActionConnectors;
};
exports.getRuleActionConnectorsForExport = getRuleActionConnectorsForExport;