"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createPrebuiltRuleAssetsClient = void 0;
var _lodash = require("lodash");
var _invariant = require("../../../../../../common/utils/invariant");
var _with_security_span = require("../../../../../utils/with_security_span");
var _prebuilt_rule_assets_validation = require("./prebuilt_rule_assets_validation");
var _prebuilt_rule_assets_type = require("./prebuilt_rule_assets_type");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const MAX_PREBUILT_RULES_COUNT = 10_000;
const MAX_ASSETS_PER_BULK_CREATE_REQUEST = 500;
const createPrebuiltRuleAssetsClient = savedObjectsClient => {
  return {
    fetchLatestAssets: () => {
      return (0, _with_security_span.withSecuritySpan)('IPrebuiltRuleAssetsClient.fetchLatestAssets', async () => {
        var _findResult$aggregati, _findResult$aggregati2, _findResult$aggregati3;
        const findResult = await savedObjectsClient.find({
          type: _prebuilt_rule_assets_type.PREBUILT_RULE_ASSETS_SO_TYPE,
          aggs: {
            rules: {
              terms: {
                field: `${_prebuilt_rule_assets_type.PREBUILT_RULE_ASSETS_SO_TYPE}.attributes.rule_id`,
                size: MAX_PREBUILT_RULES_COUNT
              },
              aggs: {
                latest_version: {
                  top_hits: {
                    size: 1,
                    sort: {
                      [`${_prebuilt_rule_assets_type.PREBUILT_RULE_ASSETS_SO_TYPE}.version`]: 'desc'
                    }
                  }
                }
              }
            }
          }
        });
        const buckets = (_findResult$aggregati = (_findResult$aggregati2 = findResult.aggregations) === null || _findResult$aggregati2 === void 0 ? void 0 : (_findResult$aggregati3 = _findResult$aggregati2.rules) === null || _findResult$aggregati3 === void 0 ? void 0 : _findResult$aggregati3.buckets) !== null && _findResult$aggregati !== void 0 ? _findResult$aggregati : [];
        (0, _invariant.invariant)(Array.isArray(buckets), 'Expected buckets to be an array');
        const ruleAssets = buckets.map(bucket => {
          const hit = bucket.latest_version.hits.hits[0];
          return hit._source[_prebuilt_rule_assets_type.PREBUILT_RULE_ASSETS_SO_TYPE];
        });
        return (0, _prebuilt_rule_assets_validation.validatePrebuiltRuleAssets)(ruleAssets);
      });
    },
    fetchLatestVersions: () => {
      return (0, _with_security_span.withSecuritySpan)('IPrebuiltRuleAssetsClient.fetchLatestVersions', async () => {
        var _findResult$aggregati4, _findResult$aggregati5, _findResult$aggregati6;
        const findResult = await savedObjectsClient.find({
          type: _prebuilt_rule_assets_type.PREBUILT_RULE_ASSETS_SO_TYPE,
          aggs: {
            rules: {
              terms: {
                field: `${_prebuilt_rule_assets_type.PREBUILT_RULE_ASSETS_SO_TYPE}.attributes.rule_id`,
                size: MAX_PREBUILT_RULES_COUNT
              },
              aggs: {
                latest_version: {
                  top_hits: {
                    size: 1,
                    sort: [{
                      [`${_prebuilt_rule_assets_type.PREBUILT_RULE_ASSETS_SO_TYPE}.version`]: 'desc'
                    }],
                    _source: [`${_prebuilt_rule_assets_type.PREBUILT_RULE_ASSETS_SO_TYPE}.rule_id`, `${_prebuilt_rule_assets_type.PREBUILT_RULE_ASSETS_SO_TYPE}.version`]
                  }
                }
              }
            }
          }
        });
        const buckets = (_findResult$aggregati4 = (_findResult$aggregati5 = findResult.aggregations) === null || _findResult$aggregati5 === void 0 ? void 0 : (_findResult$aggregati6 = _findResult$aggregati5.rules) === null || _findResult$aggregati6 === void 0 ? void 0 : _findResult$aggregati6.buckets) !== null && _findResult$aggregati4 !== void 0 ? _findResult$aggregati4 : [];
        (0, _invariant.invariant)(Array.isArray(buckets), 'Expected buckets to be an array');
        return buckets.map(bucket => {
          const hit = bucket.latest_version.hits.hits[0];
          const soAttributes = hit._source[_prebuilt_rule_assets_type.PREBUILT_RULE_ASSETS_SO_TYPE];
          const versionInfo = {
            rule_id: soAttributes.rule_id,
            version: soAttributes.version
          };
          return versionInfo;
        });
      });
    },
    fetchAssetsByVersionInfo: versions => {
      return (0, _with_security_span.withSecuritySpan)('IPrebuiltRuleAssetsClient.fetchAssetsByVersionInfo', async () => {
        if (versions.length === 0) {
          // NOTE: without early return it would build incorrect filter and fetch all existing saved objects
          return [];
        }
        const attr = `${_prebuilt_rule_assets_type.PREBUILT_RULE_ASSETS_SO_TYPE}.attributes`;
        const filter = versions.map(v => `(${attr}.rule_id: ${v.rule_id} AND ${attr}.version: ${v.version})`).join(' OR ');
        const findResult = await savedObjectsClient.find({
          type: _prebuilt_rule_assets_type.PREBUILT_RULE_ASSETS_SO_TYPE,
          filter,
          perPage: MAX_PREBUILT_RULES_COUNT
        });
        const ruleAssets = findResult.saved_objects.map(so => so.attributes);
        return (0, _prebuilt_rule_assets_validation.validatePrebuiltRuleAssets)(ruleAssets);
      });
    },
    bulkCreateAssets: assets => {
      return (0, _with_security_span.withSecuritySpan)('IPrebuiltRuleAssetsClient.bulkCreateAssets', async () => {
        const validAssets = (0, _prebuilt_rule_assets_validation.validatePrebuiltRuleAssets)(assets);
        const bulkCreateObjects = validAssets.map(asset => ({
          id: `${asset.rule_id}_${asset.version}`,
          type: _prebuilt_rule_assets_type.PREBUILT_RULE_ASSETS_SO_TYPE,
          attributes: asset
        }));
        const bulkCreateChunks = (0, _lodash.chunk)(bulkCreateObjects, MAX_ASSETS_PER_BULK_CREATE_REQUEST);
        for (const chunkOfObjects of bulkCreateChunks) {
          await savedObjectsClient.bulkCreate(chunkOfObjects, {
            refresh: false,
            overwrite: true
          });
        }
      });
    }
  };
};
exports.createPrebuiltRuleAssetsClient = createPrebuiltRuleAssetsClient;