"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.convertRuleToDiffable = void 0;
var _constants = require("../../../../../../../common/constants");
var _utility_types = require("../../../../../../../common/utility_types");
var _extract_building_block_object = require("./extract_building_block_object");
var _extract_rule_data_query = require("./extract_rule_data_query");
var _extract_rule_data_source = require("./extract_rule_data_source");
var _extract_rule_name_override_object = require("./extract_rule_name_override_object");
var _extract_rule_schedule = require("./extract_rule_schedule");
var _extract_timeline_template_reference = require("./extract_timeline_template_reference");
var _extract_timestamp_override_object = require("./extract_timestamp_override_object");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Normalizes a given rule to the form which is suitable for passing to the diff algorithm.
 * Read more in the JSDoc description of DiffableRule.
 */
const convertRuleToDiffable = rule => {
  const commonFields = extractDiffableCommonFields(rule);
  switch (rule.type) {
    case 'query':
      return {
        ...commonFields,
        ...extractDiffableCustomQueryFields(rule)
      };
    case 'saved_query':
      return {
        ...commonFields,
        ...extractDiffableSavedQueryFieldsFromRuleObject(rule)
      };
    case 'eql':
      return {
        ...commonFields,
        ...extractDiffableEqlFieldsFromRuleObject(rule)
      };
    case 'threat_match':
      return {
        ...commonFields,
        ...extractDiffableThreatMatchFieldsFromRuleObject(rule)
      };
    case 'threshold':
      return {
        ...commonFields,
        ...extractDiffableThresholdFieldsFromRuleObject(rule)
      };
    case 'machine_learning':
      return {
        ...commonFields,
        ...extractDiffableMachineLearningFieldsFromRuleObject(rule)
      };
    case 'new_terms':
      return {
        ...commonFields,
        ...extractDiffableNewTermsFieldsFromRuleObject(rule)
      };
    default:
      return (0, _utility_types.assertUnreachable)(rule, 'Unhandled rule type');
  }
};
exports.convertRuleToDiffable = convertRuleToDiffable;
const extractDiffableCommonFields = rule => {
  var _rule$meta, _rule$tags, _rule$severity_mappin, _rule$risk_score_mapp, _rule$references, _rule$false_positives, _rule$threat, _rule$note, _rule$setup, _rule$related_integra, _rule$required_fields, _rule$author, _rule$license, _rule$actions, _rule$throttle, _rule$exceptions_list, _rule$max_signals;
  return {
    // --------------------- REQUIRED FIELDS
    // Technical fields
    rule_id: rule.rule_id,
    version: rule.version,
    meta: (_rule$meta = rule.meta) !== null && _rule$meta !== void 0 ? _rule$meta : {},
    // Main domain fields
    name: rule.name,
    tags: (_rule$tags = rule.tags) !== null && _rule$tags !== void 0 ? _rule$tags : [],
    description: rule.description,
    severity: rule.severity,
    severity_mapping: (_rule$severity_mappin = rule.severity_mapping) !== null && _rule$severity_mappin !== void 0 ? _rule$severity_mappin : [],
    risk_score: rule.risk_score,
    risk_score_mapping: (_rule$risk_score_mapp = rule.risk_score_mapping) !== null && _rule$risk_score_mapp !== void 0 ? _rule$risk_score_mapp : [],
    // About -> Advanced settings
    references: (_rule$references = rule.references) !== null && _rule$references !== void 0 ? _rule$references : [],
    false_positives: (_rule$false_positives = rule.false_positives) !== null && _rule$false_positives !== void 0 ? _rule$false_positives : [],
    threat: (_rule$threat = rule.threat) !== null && _rule$threat !== void 0 ? _rule$threat : [],
    note: (_rule$note = rule.note) !== null && _rule$note !== void 0 ? _rule$note : '',
    setup: (_rule$setup = rule.setup) !== null && _rule$setup !== void 0 ? _rule$setup : '',
    related_integrations: (_rule$related_integra = rule.related_integrations) !== null && _rule$related_integra !== void 0 ? _rule$related_integra : [],
    required_fields: (_rule$required_fields = rule.required_fields) !== null && _rule$required_fields !== void 0 ? _rule$required_fields : [],
    author: (_rule$author = rule.author) !== null && _rule$author !== void 0 ? _rule$author : [],
    license: (_rule$license = rule.license) !== null && _rule$license !== void 0 ? _rule$license : '',
    // Other domain fields
    rule_schedule: (0, _extract_rule_schedule.extractRuleSchedule)(rule),
    actions: (_rule$actions = rule.actions) !== null && _rule$actions !== void 0 ? _rule$actions : [],
    throttle: (_rule$throttle = rule.throttle) !== null && _rule$throttle !== void 0 ? _rule$throttle : 'no_actions',
    exceptions_list: (_rule$exceptions_list = rule.exceptions_list) !== null && _rule$exceptions_list !== void 0 ? _rule$exceptions_list : [],
    max_signals: (_rule$max_signals = rule.max_signals) !== null && _rule$max_signals !== void 0 ? _rule$max_signals : _constants.DEFAULT_MAX_SIGNALS,
    // --------------------- OPTIONAL FIELDS
    rule_name_override: (0, _extract_rule_name_override_object.extractRuleNameOverrideObject)(rule),
    timestamp_override: (0, _extract_timestamp_override_object.extractTimestampOverrideObject)(rule),
    timeline_template: (0, _extract_timeline_template_reference.extractTimelineTemplateReference)(rule),
    building_block: (0, _extract_building_block_object.extractBuildingBlockObject)(rule)
  };
};
const extractDiffableCustomQueryFields = rule => {
  return {
    type: rule.type,
    data_query: (0, _extract_rule_data_query.extractRuleKqlQuery)(rule.query, rule.language, rule.filters, rule.saved_id),
    data_source: (0, _extract_rule_data_source.extractRuleDataSource)(rule.index, rule.data_view_id),
    alert_suppression: rule.alert_suppression
  };
};
const extractDiffableSavedQueryFieldsFromRuleObject = rule => {
  return {
    type: rule.type,
    data_query: (0, _extract_rule_data_query.extractRuleKqlQuery)(rule.query, rule.language, rule.filters, rule.saved_id),
    data_source: (0, _extract_rule_data_source.extractRuleDataSource)(rule.index, rule.data_view_id),
    alert_suppression: rule.alert_suppression
  };
};
const extractDiffableEqlFieldsFromRuleObject = rule => {
  return {
    type: rule.type,
    data_query: (0, _extract_rule_data_query.extractRuleEqlQuery)(rule.query, rule.language, rule.filters),
    data_source: (0, _extract_rule_data_source.extractRuleDataSource)(rule.index, rule.data_view_id),
    event_category_override: rule.event_category_override,
    timestamp_field: rule.timestamp_field,
    tiebreaker_field: rule.tiebreaker_field
  };
};
const extractDiffableThreatMatchFieldsFromRuleObject = rule => {
  return {
    type: rule.type,
    data_query: (0, _extract_rule_data_query.extractRuleKqlQuery)(rule.query, rule.language, rule.filters, rule.saved_id),
    data_source: (0, _extract_rule_data_source.extractRuleDataSource)(rule.index, rule.data_view_id),
    threat_query: (0, _extract_rule_data_query.extractInlineKqlQuery)(rule.threat_query, rule.threat_language, rule.threat_filters),
    threat_index: rule.threat_index,
    threat_mapping: rule.threat_mapping,
    threat_indicator_path: rule.threat_indicator_path,
    concurrent_searches: rule.concurrent_searches,
    items_per_search: rule.items_per_search
  };
};
const extractDiffableThresholdFieldsFromRuleObject = rule => {
  return {
    type: rule.type,
    data_query: (0, _extract_rule_data_query.extractRuleKqlQuery)(rule.query, rule.language, rule.filters, rule.saved_id),
    data_source: (0, _extract_rule_data_source.extractRuleDataSource)(rule.index, rule.data_view_id),
    threshold: rule.threshold
  };
};
const extractDiffableMachineLearningFieldsFromRuleObject = rule => {
  return {
    type: rule.type,
    machine_learning_job_id: rule.machine_learning_job_id,
    anomaly_threshold: rule.anomaly_threshold
  };
};
const extractDiffableNewTermsFieldsFromRuleObject = rule => {
  return {
    type: rule.type,
    data_query: (0, _extract_rule_data_query.extractInlineKqlQuery)(rule.query, rule.language, rule.filters),
    data_source: (0, _extract_rule_data_source.extractRuleDataSource)(rule.index, rule.data_view_id),
    new_terms_fields: rule.new_terms_fields,
    history_window_start: rule.history_window_start
  };
};