"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.calculateRuleDiff = void 0;
var _three_way_diff = require("../../../../../../common/detection_engine/prebuilt_rules/model/diff/three_way_diff/three_way_diff");
var _calculate_rule_fields_diff = require("./calculation/calculate_rule_fields_diff");
var _convert_rule_to_diffable = require("./normalization/convert_rule_to_diffable");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Calculates a rule diff for a given set of 3 versions of the rule:
 *   - currenly installed version
 *   - base version that is the corresponding stock rule content
 *   - target version which is the stock rule content the user wants to update the rule to
 */
const calculateRuleDiff = args => {
  /*
    1. Convert current, base and target versions to `DiffableRule`.
    2. Calculate a `RuleFieldsDiff`. For every top-level field of `DiffableRule`:
      2.1. Pick a code path based on the rule type.
      2.2. Pick a concrete diff algorithm (function) per rule field based on the field name or type.
        - one algo for rule name and other simple string fields
        - another one for tags and other arrays of keywords
        - another one for multiline text fields (investigation guide, setup guide, etc)
        - another one for `data_source`
        - etc
      2.3. Call the picked diff function to get a `ThreeWayDiff` result
      2.4. Add the result to the `RuleFieldsDiff` object as a key-value pair "fieldName: ThreeWayDiff".
    3. Create and return a result based on the `RuleFieldsDiff`.
  */

  const {
    baseVersion,
    currentVersion,
    targetVersion
  } = args;
  const diffableBaseVersion = baseVersion ? (0, _convert_rule_to_diffable.convertRuleToDiffable)(baseVersion) : undefined;
  const diffableCurrentVersion = (0, _convert_rule_to_diffable.convertRuleToDiffable)(currentVersion);
  const diffableTargetVersion = (0, _convert_rule_to_diffable.convertRuleToDiffable)(targetVersion);
  const fieldsDiff = (0, _calculate_rule_fields_diff.calculateRuleFieldsDiff)({
    base_version: diffableBaseVersion || _three_way_diff.MissingVersion,
    current_version: diffableCurrentVersion,
    target_version: diffableTargetVersion
  });
  const hasAnyFieldConflict = Object.values(fieldsDiff).some(fieldDiff => fieldDiff.has_conflict);
  return {
    ruleDiff: {
      fields: fieldsDiff,
      has_conflict: hasAnyFieldConflict
    },
    ruleVersions: {
      input: {
        current: currentVersion,
        base: baseVersion,
        target: targetVersion
      },
      output: {
        current: diffableCurrentVersion,
        base: diffableBaseVersion,
        target: diffableTargetVersion
      }
    }
  };
};
exports.calculateRuleDiff = calculateRuleDiff;