"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.reviewRuleUpgradeRoute = void 0;
var _lodash = require("lodash");
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _prebuilt_rules = require("../../../../../../common/detection_engine/prebuilt_rules");
var _calculate_rule_diff = require("../../logic/diff/calculate_rule_diff");
var _utils = require("../../../routes/utils");
var _prebuilt_rule_assets_client = require("../../logic/rule_assets/prebuilt_rule_assets_client");
var _prebuilt_rule_objects_client = require("../../logic/rule_objects/prebuilt_rule_objects_client");
var _get_version_buckets = require("../../model/rule_versions/get_version_buckets");
var _invariant = require("../../../../../../common/utils/invariant");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const reviewRuleUpgradeRoute = router => {
  router.post({
    path: _prebuilt_rules.REVIEW_RULE_UPGRADE_URL,
    validate: {},
    options: {
      tags: ['access:securitySolution']
    }
  }, async (context, request, response) => {
    const siemResponse = (0, _utils.buildSiemResponse)(response);
    try {
      const ctx = await context.resolve(['core', 'alerting']);
      const soClient = ctx.core.savedObjects.client;
      const rulesClient = ctx.alerting.getRulesClient();
      const ruleAssetsClient = (0, _prebuilt_rule_assets_client.createPrebuiltRuleAssetsClient)(soClient);
      const ruleObjectsClient = (0, _prebuilt_rule_objects_client.createPrebuiltRuleObjectsClient)(rulesClient);
      const [latestVersions, {
        installedVersions,
        installedRules
      }] = await Promise.all([ruleAssetsClient.fetchLatestVersions(), ruleObjectsClient.fetchInstalledRules()]);
      const versionBuckets = (0, _get_version_buckets.getVersionBuckets)({
        latestVersions,
        installedVersions
      });
      const [baseRules, latestRules] = await Promise.all([ruleAssetsClient.fetchAssetsByVersionInfo(versionBuckets.installedVersionsToUpgrade), ruleAssetsClient.fetchAssetsByVersionInfo(versionBuckets.latestVersionsToUpgrade)]);
      const ruleDiffCalculationArgs = getRuleDiffCalculationArgs(versionBuckets.installedVersionsToUpgrade, installedRules, baseRules, latestRules);
      const ruleDiffCalculationResults = ruleDiffCalculationArgs.map(args => {
        return (0, _calculate_rule_diff.calculateRuleDiff)(args);
      });
      const body = {
        status_code: 200,
        message: 'OK',
        attributes: {
          stats: calculateRuleStats(ruleDiffCalculationResults),
          rules: calculateRuleInfos(ruleDiffCalculationResults)
        }
      };
      return response.ok({
        body
      });
    } catch (err) {
      const error = (0, _securitysolutionEsUtils.transformError)(err);
      return siemResponse.error({
        body: error.message,
        statusCode: error.statusCode
      });
    }
  });
};
exports.reviewRuleUpgradeRoute = reviewRuleUpgradeRoute;
const getRuleDiffCalculationArgs = (installedVersionsToUpgrade, installedRules, baseRules, latestRules) => {
  const installedRulesMap = new Map(installedRules.map(r => [r.rule_id, r]));
  const baseRulesMap = new Map(baseRules.map(r => [r.rule_id, r]));
  const latestRulesMap = new Map(latestRules.map(r => [r.rule_id, r]));
  const result = [];
  installedVersionsToUpgrade.forEach(versionToUpgrade => {
    const ruleId = versionToUpgrade.rule_id;
    const installedRule = installedRulesMap.get(ruleId);
    const baseRule = baseRulesMap.get(ruleId);
    const latestRule = latestRulesMap.get(ruleId);

    // baseRule can be undefined if the rule has no historical versions, but other versions should always be present
    (0, _invariant.invariant)(installedRule != null, `installedRule is not found for rule_id: ${ruleId}`);
    (0, _invariant.invariant)(latestRule != null, `latestRule is not found for rule_id: ${ruleId}`);
    result.push({
      currentVersion: installedRule,
      baseVersion: baseRule,
      targetVersion: latestRule
    });
  });
  return result;
};
const calculateRuleStats = results => {
  const allTags = new Set(results.flatMap(result => result.ruleVersions.input.current.tags));
  return {
    num_rules_to_upgrade_total: results.length,
    tags: [...allTags].sort((a, b) => a.localeCompare(b))
  };
};
const calculateRuleInfos = results => {
  return results.map(result => {
    const {
      ruleDiff,
      ruleVersions
    } = result;
    const installedCurrentVersion = ruleVersions.input.current;
    const diffableCurrentVersion = ruleVersions.output.current;
    return {
      id: installedCurrentVersion.id,
      rule_id: installedCurrentVersion.rule_id,
      rule: diffableCurrentVersion,
      diff: {
        fields: (0, _lodash.pickBy)(ruleDiff.fields, fieldDiff => fieldDiff.has_update || fieldDiff.has_conflict),
        has_conflict: ruleDiff.has_conflict
      }
    };
  });
};