"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ActionCreateService = void 0;
var _uuid = require("uuid");
var _moment = _interopRequireDefault(require("moment"));
var _common = require("@kbn/fleet-plugin/common");
var _common2 = require("@kbn/cases-plugin/common");
var _constants = require("../../../../../common/constants");
var _constants2 = require("../../../../../common/endpoint/service/response_actions/constants");
var _constants3 = require("../../../../../common/endpoint/constants");
var _utils = require("../../../utils");
var _ = require("..");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const commandToFeatureKeyMap = new Map([['isolate', 'HOST_ISOLATION'], ['unisolate', 'HOST_ISOLATION'], ['kill-process', 'KILL_PROCESS'], ['suspend-process', 'SUSPEND_PROCESS'], ['running-processes', 'RUNNING_PROCESSES'], ['get-file', 'GET_FILE'], ['execute', 'EXECUTE']]);
const returnActionIdCommands = ['isolate', 'unisolate'];
class ActionCreateService {
  constructor(esClient, endpointContext) {
    this.esClient = esClient;
    this.endpointContext = endpointContext;
  }
  async createAction(payload, casesClient) {
    var _payload$comment, _getActionParameters;
    const featureKey = commandToFeatureKeyMap.get(payload.command);
    if (featureKey) {
      this.endpointContext.service.getFeatureUsageService().notifyUsage(featureKey);
    }
    const logger = this.endpointContext.logFactory.get('hostIsolation');

    // fetch the Agent IDs to send the commands to
    const endpointIDs = [...new Set(payload.endpoint_ids)]; // dedupe
    const endpointData = await this.endpointContext.service.getEndpointMetadataService().getMetadataForEndpoints(this.esClient, endpointIDs);
    const agents = endpointData.map(endpoint => endpoint.elastic.agent.id);

    // create an Action ID and dispatch it to ES & Fleet Server
    const actionID = (0, _uuid.v4)();
    let fleetActionIndexResult;
    let logsEndpointActionsResult;
    const getActionParameters = () => {
      var _payload$parameters;
      // set timeout to 4h (if not specified or when timeout is specified as 0) when command is `execute`
      if (payload.command === 'execute') {
        const actionRequestParams = payload.parameters;
        if (typeof (actionRequestParams === null || actionRequestParams === void 0 ? void 0 : actionRequestParams.timeout) === 'undefined') {
          return {
            ...actionRequestParams,
            timeout: _constants2.DEFAULT_EXECUTE_ACTION_TIMEOUT
          };
        }
        return actionRequestParams;
      }

      // for all other commands return the parameters as is
      return (_payload$parameters = payload.parameters) !== null && _payload$parameters !== void 0 ? _payload$parameters : undefined;
    };
    const doc = {
      '@timestamp': (0, _moment.default)().toISOString(),
      agent: {
        id: agents
      },
      EndpointActions: {
        action_id: actionID,
        expiration: (0, _moment.default)().add(2, 'weeks').toISOString(),
        type: 'INPUT_ACTION',
        input_type: 'endpoint',
        data: {
          command: payload.command,
          comment: (_payload$comment = payload.comment) !== null && _payload$comment !== void 0 ? _payload$comment : undefined,
          ...(payload.alert_ids ? {
            alert_id: payload.alert_ids
          } : {}),
          parameters: (_getActionParameters = getActionParameters()) !== null && _getActionParameters !== void 0 ? _getActionParameters : undefined
        }
      },
      user: {
        id: payload.user ? payload.user.username : 'unknown'
      },
      ...(payload.rule_id && payload.rule_name ? {
        rule: {
          id: payload.rule_id,
          name: payload.rule_name
        }
      } : {})
    };

    // if .logs-endpoint.actions data stream exists
    // try to create action request record in .logs-endpoint.actions DS as the current user
    // (from >= v7.16, use this check to ensure the current user has privileges to write to the new index)
    // and allow only users with superuser privileges to write to fleet indices
    // const logger = endpointContext.logFactory.get('host-isolation');
    const doesLogsEndpointActionsDsExist = await (0, _utils.doLogsEndpointActionDsExists)({
      esClient: this.esClient,
      logger,
      dataStreamName: _constants3.ENDPOINT_ACTIONS_DS
    });

    // if the new endpoint indices/data streams exists
    // write the action request to the new endpoint index
    if (doesLogsEndpointActionsDsExist) {
      logsEndpointActionsResult = await this.esClient.index({
        index: _constants3.ENDPOINT_ACTIONS_INDEX,
        body: {
          ...doc
        },
        refresh: 'wait_for'
      }, {
        meta: true
      });
      if (logsEndpointActionsResult.statusCode !== 201) {
        throw new Error(logsEndpointActionsResult.body.result);
      }
    }

    // add signature to doc
    const fleetActionDoc = {
      ...doc.EndpointActions,
      '@timestamp': doc['@timestamp'],
      agents,
      timeout: 300,
      // 5 minutes
      user_id: doc.user.id
    };
    const fleetActionDocSignature = await this.endpointContext.service.getMessageSigningService().sign(fleetActionDoc);
    const signedFleetActionDoc = {
      ...fleetActionDoc,
      signed: {
        data: fleetActionDocSignature.data.toString('base64'),
        signature: fleetActionDocSignature.signature
      }
    };

    // write actions to .fleet-actions index
    try {
      fleetActionIndexResult = await this.esClient.index({
        index: _common.AGENT_ACTIONS_INDEX,
        body: signedFleetActionDoc,
        refresh: 'wait_for'
      }, {
        meta: true
      });
      if (fleetActionIndexResult.statusCode !== 201) {
        throw new Error(fleetActionIndexResult.body.result);
      }
    } catch (e) {
      // create entry in .logs-endpoint.action.responses-default data stream
      // when writing to .fleet-actions fails
      if (doesLogsEndpointActionsDsExist) {
        await createFailedActionResponseEntry({
          esClient: this.esClient,
          doc: {
            '@timestamp': (0, _moment.default)().toISOString(),
            agent: doc.agent,
            EndpointActions: {
              action_id: doc.EndpointActions.action_id,
              completed_at: (0, _moment.default)().toISOString(),
              started_at: (0, _moment.default)().toISOString(),
              data: doc.EndpointActions.data
            }
          },
          logger
        });
      }
      throw e;
    }
    if (casesClient) {
      var _payload$case_ids;
      // convert any alert IDs into cases
      let caseIDs = ((_payload$case_ids = payload.case_ids) === null || _payload$case_ids === void 0 ? void 0 : _payload$case_ids.slice()) || [];
      if (payload.alert_ids && payload.alert_ids.length > 0) {
        const newIDs = await Promise.all(payload.alert_ids.map(async alertID => {
          const cases = await casesClient.cases.getCasesByAlertID({
            alertID,
            options: {
              owner: _constants.APP_ID
            }
          });
          return cases.map(caseInfo => {
            return caseInfo.id;
          });
        }));
        caseIDs = caseIDs.concat(...newIDs);
      }
      caseIDs = [...new Set(caseIDs)];

      // Update all cases with a comment
      if (caseIDs.length > 0) {
        const targets = endpointData.map(endpt => ({
          hostname: endpt.host.hostname,
          endpointId: endpt.agent.id
        }));
        await Promise.all(caseIDs.map(caseId => casesClient.attachments.add({
          caseId,
          comment: {
            type: _common2.CommentType.actions,
            comment: payload.comment || '',
            actions: {
              targets,
              type: payload.command
            },
            owner: _constants.APP_ID
          }
        })));
      }
    }
    const actionId = returnActionIdCommands.includes(payload.command) ? {
      action: actionID
    } : {};
    const data = await (0, _.getActionDetailsById)(this.esClient, this.endpointContext.service.getEndpointMetadataService(), actionID);
    return {
      ...actionId,
      ...data
    };
  }
}
exports.ActionCreateService = ActionCreateService;
const createFailedActionResponseEntry = async ({
  esClient,
  doc,
  logger
}) => {
  try {
    await esClient.index({
      index: `${_constants3.ENDPOINT_ACTION_RESPONSES_DS}-default`,
      body: {
        ...doc,
        error: {
          code: _constants3.failedFleetActionErrorCode,
          message: 'Failed to deliver action request to fleet'
        }
      }
    });
  } catch (e) {
    logger.error(e);
  }
};