"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.upsertTimelineColumn = exports.upsertTableColumn = exports.updateTimelineTitleAndDescription = exports.updateTimelineSort = exports.updateTimelineShowTimeline = exports.updateTimelineSessionViewConfig = exports.updateTimelineRange = exports.updateTimelineProviders = exports.updateTimelineProviderType = exports.updateTimelineProviderProperties = exports.updateTimelineProviderExcluded = exports.updateTimelineProviderEnabled = exports.updateTimelinePerPageOptions = exports.updateTimelineKqlMode = exports.updateTimelineItemsPerPage = exports.updateTimelineIsFavorite = exports.updateTimelineGraphEventId = exports.updateTimelineEventType = exports.updateTimelineDetailsPanel = exports.updateTimelineColumns = exports.updateTableSort = exports.updateTableColumns = exports.updateSavedQuery = exports.updateFilters = exports.updateExcludedRowRenderersIds = exports.unPinTimelineEvent = exports.shouldResetActiveTimelineContext = exports.setSelectedTableEvents = exports.setLoadingTableEvents = exports.setInitializeTimelineSettings = exports.setDeletedTableEvents = exports.removeTimelineProvider = exports.removeTimelineColumn = exports.removeTableColumn = exports.pinTimelineEvent = exports.isNotNull = exports.applyKqlFilterQueryDraft = exports.applyDeltaToTimelineColumnWidth = exports.applyDeltaToTableColumnWidth = exports.addTimelineToStore = exports.addTimelineProviders = exports.addTimelineNoteToEvent = exports.addTimelineNote = exports.addNewTimeline = void 0;
var _fp = require("lodash/fp");
var _uuid = require("uuid");
var _data_provider = require("../../components/timeline/data_providers/data_provider");
var _timeline = require("../../../../common/types/timeline");
var _normalize_time_range = require("../../../common/utils/normalize_time_range");
var _defaults = require("./defaults");
var _default_date_settings = require("../../../common/utils/default_date_settings");
var _constants = require("../../components/timeline/body/constants");
var _active_timeline_context = require("../../containers/active_timeline_context");
var _helpers = require("../../components/timeline/data_providers/helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const isNotNull = value => value !== null;
exports.isNotNull = isNotNull;
const addTimelineNote = ({
  id,
  noteId,
  timelineById
}) => {
  const timeline = timelineById[id];
  return {
    ...timelineById,
    [id]: {
      ...timeline,
      noteIds: [...timeline.noteIds, noteId]
    }
  };
};
exports.addTimelineNote = addTimelineNote;
const addTimelineNoteToEvent = ({
  id,
  noteId,
  eventId,
  timelineById
}) => {
  const timeline = timelineById[id];
  const existingNoteIds = (0, _fp.getOr)([], `eventIdToNoteIds.${eventId}`, timeline);
  return {
    ...timelineById,
    [id]: {
      ...timeline,
      eventIdToNoteIds: {
        ...timeline.eventIdToNoteIds,
        ...{
          [eventId]: (0, _fp.uniq)([...existingNoteIds, noteId])
        }
      }
    }
  };
};
exports.addTimelineNoteToEvent = addTimelineNoteToEvent;
const shouldResetActiveTimelineContext = (id, oldTimeline, newTimeline) => {
  if (id === _timeline.TimelineId.active && oldTimeline.savedObjectId !== newTimeline.savedObjectId) {
    return true;
  }
  return false;
};

/**
 * Add a saved object timeline to the store
 * and default the value to what need to be if values are null
 */
exports.shouldResetActiveTimelineContext = shouldResetActiveTimelineContext;
const addTimelineToStore = ({
  id,
  resolveTimelineConfig,
  timeline,
  timelineById
}) => {
  if (shouldResetActiveTimelineContext(id, timelineById[id], timeline)) {
    _active_timeline_context.activeTimeline.setActivePage(0);
    _active_timeline_context.activeTimeline.setExpandedDetail({});
  }
  return {
    ...timelineById,
    [id]: {
      ...timeline,
      filterManager: timelineById[id].filterManager,
      isLoading: timelineById[id].isLoading,
      initialized: timelineById[id].initialized,
      resolveTimelineConfig,
      dateRange: timeline.status === _timeline.TimelineStatus.immutable && timeline.timelineType === _timeline.TimelineType.template ? {
        start: _default_date_settings.DEFAULT_FROM_MOMENT.toISOString(),
        end: _default_date_settings.DEFAULT_TO_MOMENT.toISOString()
      } : timeline.dateRange
    }
  };
};
exports.addTimelineToStore = addTimelineToStore;
/** Adds a new `Timeline` to the provided collection of `TimelineById` */
const addNewTimeline = ({
  id,
  timelineById,
  timelineType,
  dateRange: maybeDateRange,
  ...timelineProps
}) => {
  const timeline = timelineById[id];
  const {
    from: startDateRange,
    to: endDateRange
  } = (0, _normalize_time_range.normalizeTimeRange)({
    from: '',
    to: ''
  });
  const dateRange = maybeDateRange !== null && maybeDateRange !== void 0 ? maybeDateRange : {
    start: startDateRange,
    end: endDateRange
  };
  const templateTimelineInfo = timelineType === _timeline.TimelineType.template ? {
    templateTimelineId: (0, _uuid.v4)(),
    templateTimelineVersion: 1
  } : {};
  return {
    ...timelineById,
    [id]: {
      id,
      ...(timeline ? timeline : {}),
      ..._defaults.timelineDefaults,
      ...timelineProps,
      dateRange,
      savedObjectId: null,
      version: null,
      isSaving: false,
      isLoading: false,
      timelineType,
      ...templateTimelineInfo
    }
  };
};
exports.addNewTimeline = addNewTimeline;
const pinTimelineEvent = ({
  id,
  eventId,
  timelineById
}) => {
  const timeline = timelineById[id];
  return {
    ...timelineById,
    [id]: {
      ...timeline,
      pinnedEventIds: {
        ...timeline.pinnedEventIds,
        ...{
          [eventId]: true
        }
      }
    }
  };
};
exports.pinTimelineEvent = pinTimelineEvent;
const updateTimelineShowTimeline = ({
  id,
  show,
  timelineById
}) => {
  const timeline = timelineById[id];
  return {
    ...timelineById,
    [id]: {
      ...timeline,
      show
    }
  };
};
exports.updateTimelineShowTimeline = updateTimelineShowTimeline;
const updateTimelineGraphEventId = ({
  id,
  graphEventId,
  timelineById
}) => {
  const timeline = timelineById[id];
  return {
    ...timelineById,
    [id]: {
      ...timeline,
      graphEventId,
      ...(graphEventId === '' && id === _timeline.TimelineId.active ? {
        activeTab: timeline.prevActiveTab,
        prevActiveTab: timeline.activeTab
      } : {})
    }
  };
};
exports.updateTimelineGraphEventId = updateTimelineGraphEventId;
const updateTimelineSessionViewConfig = ({
  id,
  sessionViewConfig,
  timelineById
}) => {
  const timeline = timelineById[id];
  return {
    ...timelineById,
    [id]: {
      ...timeline,
      sessionViewConfig
    }
  };
};
exports.updateTimelineSessionViewConfig = updateTimelineSessionViewConfig;
const queryMatchCustomizer = (dp1, dp2) => {
  if (dp1.field === dp2.field && dp1.value === dp2.value && dp1.operator === dp2.operator) {
    return true;
  }
  return false;
};
const addAndToProvidersInTimeline = (id, providers, timeline, timelineById) => {
  if (providers.length === 0) return timelineById;
  let localDataProviders = (0, _fp.cloneDeep)(timeline.dataProviders);
  providers.forEach(provider => {
    const alreadyExistsProviderIndex = localDataProviders.findIndex(p => p.id === timeline.highlightedDropAndProviderId);
    const newProvider = localDataProviders[alreadyExistsProviderIndex];
    const alreadyExistsAndProviderIndex = newProvider.and.findIndex(p => p.id === provider.id);
    const {
      and,
      ...andProvider
    } = provider;
    if ((0, _fp.isEqualWith)(queryMatchCustomizer, newProvider.queryMatch, andProvider.queryMatch) || alreadyExistsAndProviderIndex === -1 && newProvider.and.filter(itemAndProvider => (0, _fp.isEqualWith)(queryMatchCustomizer, itemAndProvider.queryMatch, andProvider.queryMatch)).length > 0) {
      return timelineById;
    }
    localDataProviders = [...localDataProviders.slice(0, alreadyExistsProviderIndex), {
      ...localDataProviders[alreadyExistsProviderIndex],
      and: alreadyExistsAndProviderIndex > -1 ? [...newProvider.and.slice(0, alreadyExistsAndProviderIndex), andProvider, ...newProvider.and.slice(alreadyExistsAndProviderIndex + 1)] : [...newProvider.and, andProvider]
    }, ...localDataProviders.slice(alreadyExistsProviderIndex + 1)];
  });
  return {
    ...timelineById,
    [id]: {
      ...timeline,
      dataProviders: localDataProviders
    }
  };
};
const addProvidersToTimeline = (id, providers, timeline, timelineById) => {
  if (providers.length === 0) return timelineById;
  let localDataProviders = (0, _fp.cloneDeep)(timeline.dataProviders);
  providers.forEach(provider => {
    const alreadyExistsAtIndex = localDataProviders.findIndex(p => p.id === provider.id);
    if (alreadyExistsAtIndex > -1 && !(0, _fp.isEmpty)(localDataProviders[alreadyExistsAtIndex].and)) {
      provider.id = `${provider.id}-${localDataProviders.filter(p => p.id === provider.id).length}`;
    }
    localDataProviders = alreadyExistsAtIndex > -1 && (0, _fp.isEmpty)(localDataProviders[alreadyExistsAtIndex].and) ? [...localDataProviders.slice(0, alreadyExistsAtIndex), provider, ...localDataProviders.slice(alreadyExistsAtIndex + 1)] : [...localDataProviders, provider];
  });
  return {
    ...timelineById,
    [id]: {
      ...timeline,
      dataProviders: localDataProviders
    }
  };
};
/**
 * Adds or updates a column. When updating a column, it will be moved to the
 * new index
 */
const upsertTimelineColumn = ({
  column,
  id,
  index,
  timelineById
}) => {
  const timeline = timelineById[id];
  const alreadyExistsAtIndex = timeline.columns.findIndex(c => c.id === column.id);
  if (alreadyExistsAtIndex !== -1) {
    // remove the existing entry and add the new one at the specified index
    const reordered = timeline.columns.filter(c => c.id !== column.id);
    reordered.splice(index, 0, column); // ⚠️ mutation

    return {
      ...timelineById,
      [id]: {
        ...timeline,
        columns: reordered
      }
    };
  }

  // add the new entry at the specified index
  const columns = [...timeline.columns];
  columns.splice(index, 0, column); // ⚠️ mutation

  return {
    ...timelineById,
    [id]: {
      ...timeline,
      columns
    }
  };
};
exports.upsertTimelineColumn = upsertTimelineColumn;
const removeTimelineColumn = ({
  id,
  columnId,
  timelineById
}) => {
  const timeline = timelineById[id];
  const columns = timeline.columns.filter(c => c.id !== columnId);
  return {
    ...timelineById,
    [id]: {
      ...timeline,
      columns
    }
  };
};
exports.removeTimelineColumn = removeTimelineColumn;
const applyDeltaToTimelineColumnWidth = ({
  id,
  columnId,
  delta,
  timelineById
}) => {
  var _timeline$columns$col;
  const timeline = timelineById[id];
  const columnIndex = timeline.columns.findIndex(c => c.id === columnId);
  if (columnIndex === -1) {
    // the column was not found
    return {
      ...timelineById,
      [id]: {
        ...timeline
      }
    };
  }
  const requestedWidth = ((_timeline$columns$col = timeline.columns[columnIndex].initialWidth) !== null && _timeline$columns$col !== void 0 ? _timeline$columns$col : _constants.DEFAULT_COLUMN_MIN_WIDTH) + delta; // raw change in width
  const initialWidth = Math.max(_constants.RESIZED_COLUMN_MIN_WITH, requestedWidth); // if the requested width is smaller than the min, use the min

  const columnWithNewWidth = {
    ...timeline.columns[columnIndex],
    initialWidth
  };
  const columns = [...timeline.columns.slice(0, columnIndex), columnWithNewWidth, ...timeline.columns.slice(columnIndex + 1)];
  return {
    ...timelineById,
    [id]: {
      ...timeline,
      columns
    }
  };
};
exports.applyDeltaToTimelineColumnWidth = applyDeltaToTimelineColumnWidth;
const addTimelineProviders = ({
  id,
  providers,
  timelineById
}) => {
  const timeline = timelineById[id];
  if (timeline.highlightedDropAndProviderId !== '') {
    return addAndToProvidersInTimeline(id, providers, timeline, timelineById);
  } else {
    return addProvidersToTimeline(id, providers, timeline, timelineById);
  }
};
exports.addTimelineProviders = addTimelineProviders;
const applyKqlFilterQueryDraft = ({
  id,
  filterQuery,
  timelineById
}) => {
  const timeline = timelineById[id];
  return {
    ...timelineById,
    [id]: {
      ...timeline,
      kqlQuery: {
        ...timeline.kqlQuery,
        filterQuery
      }
    }
  };
};
exports.applyKqlFilterQueryDraft = applyKqlFilterQueryDraft;
const updateTimelineKqlMode = ({
  id,
  kqlMode,
  timelineById
}) => {
  const timeline = timelineById[id];
  return {
    ...timelineById,
    [id]: {
      ...timeline,
      kqlMode
    }
  };
};
exports.updateTimelineKqlMode = updateTimelineKqlMode;
const updateTimelineColumns = ({
  id,
  columns,
  timelineById
}) => {
  const timeline = timelineById[id];
  return {
    ...timelineById,
    [id]: {
      ...timeline,
      columns
    }
  };
};
exports.updateTimelineColumns = updateTimelineColumns;
const updateTimelineTitleAndDescription = ({
  description,
  id,
  title,
  timelineById
}) => {
  const timeline = timelineById[id];
  return {
    ...timelineById,
    [id]: {
      ...timeline,
      description: description.trim(),
      title: title.trim()
    }
  };
};
exports.updateTimelineTitleAndDescription = updateTimelineTitleAndDescription;
const updateTimelineEventType = ({
  id,
  eventType,
  timelineById
}) => {
  const timeline = timelineById[id];
  return {
    ...timelineById,
    [id]: {
      ...timeline,
      eventType
    }
  };
};
exports.updateTimelineEventType = updateTimelineEventType;
const updateTimelineIsFavorite = ({
  id,
  isFavorite,
  timelineById
}) => {
  const timeline = timelineById[id];
  return {
    ...timelineById,
    [id]: {
      ...timeline,
      isFavorite
    }
  };
};
exports.updateTimelineIsFavorite = updateTimelineIsFavorite;
const updateTimelineProviders = ({
  id,
  providers,
  timelineById
}) => {
  const timeline = timelineById[id];
  return {
    ...timelineById,
    [id]: {
      ...timeline,
      dataProviders: providers
    }
  };
};
exports.updateTimelineProviders = updateTimelineProviders;
const updateTimelineRange = ({
  id,
  start,
  end,
  timelineById
}) => {
  const timeline = timelineById[id];
  return {
    ...timelineById,
    [id]: {
      ...timeline,
      dateRange: {
        start,
        end
      }
    }
  };
};
exports.updateTimelineRange = updateTimelineRange;
const updateTimelineSort = ({
  id,
  sort,
  timelineById
}) => {
  const timeline = timelineById[id];
  return {
    ...timelineById,
    [id]: {
      ...timeline,
      sort
    }
  };
};
exports.updateTimelineSort = updateTimelineSort;
const updateEnabledAndProvider = (andProviderId, enabled, providerId, timeline) => timeline.dataProviders.map(provider => provider.id === providerId ? {
  ...provider,
  and: provider.and.map(andProvider => andProvider.id === andProviderId ? {
    ...andProvider,
    enabled
  } : andProvider)
} : provider);
const updateEnabledProvider = (enabled, providerId, timeline) => timeline.dataProviders.map(provider => provider.id === providerId ? {
  ...provider,
  enabled
} : provider);
const updateTimelineProviderEnabled = ({
  id,
  providerId,
  enabled,
  timelineById,
  andProviderId
}) => {
  const timeline = timelineById[id];
  return {
    ...timelineById,
    [id]: {
      ...timeline,
      dataProviders: andProviderId ? updateEnabledAndProvider(andProviderId, enabled, providerId, timeline) : updateEnabledProvider(enabled, providerId, timeline)
    }
  };
};
exports.updateTimelineProviderEnabled = updateTimelineProviderEnabled;
const updateExcludedAndProvider = (andProviderId, excluded, providerId, timeline) => timeline.dataProviders.map(provider => provider.id === providerId ? {
  ...provider,
  and: provider.and.map(andProvider => andProvider.id === andProviderId ? {
    ...andProvider,
    excluded
  } : andProvider)
} : provider);
const updateExcludedProvider = (excluded, providerId, timeline) => timeline.dataProviders.map(provider => provider.id === providerId ? {
  ...provider,
  excluded
} : provider);
const updateTimelineProviderExcluded = ({
  id,
  providerId,
  excluded,
  timelineById,
  andProviderId
}) => {
  const timeline = timelineById[id];
  return {
    ...timelineById,
    [id]: {
      ...timeline,
      dataProviders: andProviderId ? updateExcludedAndProvider(andProviderId, excluded, providerId, timeline) : updateExcludedProvider(excluded, providerId, timeline)
    }
  };
};
exports.updateTimelineProviderExcluded = updateTimelineProviderExcluded;
const updateProviderProperties = ({
  excluded,
  field,
  operator,
  providerId,
  timeline,
  value
}) => timeline.dataProviders.map(provider => provider.id === providerId ? {
  ...provider,
  excluded,
  queryMatch: {
    ...provider.queryMatch,
    field,
    displayField: field,
    value,
    displayValue: (0, _helpers.getDisplayValue)(value),
    operator
  }
} : provider);
const updateAndProviderProperties = ({
  andProviderId,
  excluded,
  field,
  operator,
  providerId,
  timeline,
  value
}) => timeline.dataProviders.map(provider => provider.id === providerId ? {
  ...provider,
  and: provider.and.map(andProvider => andProvider.id === andProviderId ? {
    ...andProvider,
    excluded,
    queryMatch: {
      ...andProvider.queryMatch,
      field,
      displayField: field,
      value,
      displayValue: (0, _helpers.getDisplayValue)(value),
      operator
    }
  } : andProvider)
} : provider);
const updateTimelineProviderProperties = ({
  andProviderId,
  excluded,
  field,
  id,
  operator,
  providerId,
  timelineById,
  value
}) => {
  const timeline = timelineById[id];
  return {
    ...timelineById,
    [id]: {
      ...timeline,
      dataProviders: andProviderId ? updateAndProviderProperties({
        andProviderId,
        excluded,
        field,
        operator,
        providerId,
        timeline,
        value
      }) : updateProviderProperties({
        excluded,
        field,
        operator,
        providerId,
        timeline,
        value
      })
    }
  };
};
exports.updateTimelineProviderProperties = updateTimelineProviderProperties;
const updateTypeAndProvider = (andProviderId, type, providerId, timeline) => timeline.dataProviders.map(provider => provider.id === providerId ? {
  ...provider,
  and: provider.and.map(andProvider => andProvider.id === andProviderId ? {
    ...andProvider,
    type,
    name: type === _data_provider.DataProviderType.template ? `${andProvider.queryMatch.field}` : '',
    queryMatch: {
      ...andProvider.queryMatch,
      displayField: undefined,
      displayValue: undefined,
      value: type === _data_provider.DataProviderType.template ? `{${andProvider.queryMatch.field}}` : '',
      operator: type === _data_provider.DataProviderType.template ? _data_provider.IS_OPERATOR : _data_provider.EXISTS_OPERATOR
    }
  } : andProvider)
} : provider);
const updateTypeProvider = (type, providerId, timeline) => timeline.dataProviders.map(provider => provider.id === providerId ? {
  ...provider,
  type,
  name: type === _data_provider.DataProviderType.template ? `${provider.queryMatch.field}` : '',
  queryMatch: {
    ...provider.queryMatch,
    displayField: undefined,
    displayValue: undefined,
    value: type === _data_provider.DataProviderType.template ? `{${provider.queryMatch.field}}` : '',
    operator: type === _data_provider.DataProviderType.template ? _data_provider.IS_OPERATOR : _data_provider.EXISTS_OPERATOR
  }
} : provider);
const updateTimelineProviderType = ({
  andProviderId,
  id,
  providerId,
  type,
  timelineById
}) => {
  const timeline = timelineById[id];
  if (timeline.timelineType !== _timeline.TimelineType.template && type === _data_provider.DataProviderType.template) {
    // Not supported, timeline template cannot have template type providers
    return timelineById;
  }
  return {
    ...timelineById,
    [id]: {
      ...timeline,
      dataProviders: andProviderId ? updateTypeAndProvider(andProviderId, type, providerId, timeline) : updateTypeProvider(type, providerId, timeline)
    }
  };
};
exports.updateTimelineProviderType = updateTimelineProviderType;
const updateTimelineItemsPerPage = ({
  id,
  itemsPerPage,
  timelineById
}) => {
  const timeline = timelineById[id];
  return {
    ...timelineById,
    [id]: {
      ...timeline,
      itemsPerPage
    }
  };
};
exports.updateTimelineItemsPerPage = updateTimelineItemsPerPage;
const updateTimelinePerPageOptions = ({
  id,
  itemsPerPageOptions,
  timelineById
}) => {
  const timeline = timelineById[id];
  return {
    ...timelineById,
    [id]: {
      ...timeline,
      itemsPerPageOptions
    }
  };
};
exports.updateTimelinePerPageOptions = updateTimelinePerPageOptions;
const removeAndProvider = (andProviderId, providerId, timeline) => {
  var _timeline$dataProvide, _timeline$dataProvide2, _timeline$dataProvide3, _timeline$dataProvide4;
  const providerIndex = timeline.dataProviders.findIndex(p => p.id === providerId);
  const providerAndIndex = (_timeline$dataProvide = timeline.dataProviders[providerIndex]) === null || _timeline$dataProvide === void 0 ? void 0 : _timeline$dataProvide.and.findIndex(p => p.id === andProviderId);
  return [...timeline.dataProviders.slice(0, providerIndex), {
    ...timeline.dataProviders[providerIndex],
    and: (_timeline$dataProvide2 = timeline.dataProviders[providerIndex]) !== null && _timeline$dataProvide2 !== void 0 && _timeline$dataProvide2.and ? [...((_timeline$dataProvide3 = timeline.dataProviders[providerIndex]) === null || _timeline$dataProvide3 === void 0 ? void 0 : _timeline$dataProvide3.and.slice(0, providerAndIndex)), ...((_timeline$dataProvide4 = timeline.dataProviders[providerIndex]) === null || _timeline$dataProvide4 === void 0 ? void 0 : _timeline$dataProvide4.and.slice(providerAndIndex + 1))] : []
  }, ...timeline.dataProviders.slice(providerIndex + 1)];
};
const removeProvider = (providerId, timeline) => {
  var _timeline$dataProvide5;
  const providerIndex = timeline.dataProviders.findIndex(p => p.id === providerId);
  return [...timeline.dataProviders.slice(0, providerIndex), ...((_timeline$dataProvide5 = timeline.dataProviders[providerIndex]) !== null && _timeline$dataProvide5 !== void 0 && _timeline$dataProvide5.and.length ? [{
    ...timeline.dataProviders[providerIndex].and.slice(0, 1)[0],
    and: [...timeline.dataProviders[providerIndex].and.slice(1)]
  }] : []), ...timeline.dataProviders.slice(providerIndex + 1)];
};
const removeTimelineProvider = ({
  id,
  providerId,
  timelineById,
  andProviderId
}) => {
  const timeline = timelineById[id];
  return {
    ...timelineById,
    [id]: {
      ...timeline,
      dataProviders: andProviderId ? removeAndProvider(andProviderId, providerId, timeline) : removeProvider(providerId, timeline)
    }
  };
};
exports.removeTimelineProvider = removeTimelineProvider;
const unPinTimelineEvent = ({
  id,
  eventId,
  timelineById
}) => {
  const timeline = timelineById[id];
  return {
    ...timelineById,
    [id]: {
      ...timeline,
      pinnedEventIds: (0, _fp.omit)(eventId, timeline.pinnedEventIds)
    }
  };
};
exports.unPinTimelineEvent = unPinTimelineEvent;
const updateSavedQuery = ({
  id,
  savedQueryId,
  timelineById
}) => {
  const timeline = timelineById[id];
  return {
    ...timelineById,
    [id]: {
      ...timeline,
      savedQueryId
    }
  };
};
exports.updateSavedQuery = updateSavedQuery;
const updateFilters = ({
  id,
  filters,
  timelineById
}) => {
  const timeline = timelineById[id];
  return {
    ...timelineById,
    [id]: {
      ...timeline,
      filters
    }
  };
};
exports.updateFilters = updateFilters;
const updateExcludedRowRenderersIds = ({
  id,
  excludedRowRendererIds,
  timelineById
}) => {
  const timeline = timelineById[id];
  return {
    ...timelineById,
    [id]: {
      ...timeline,
      excludedRowRendererIds
    }
  };
};
exports.updateExcludedRowRenderersIds = updateExcludedRowRenderersIds;
const updateTimelineDetailsPanel = action => {
  var _expandedDetails$pane;
  const {
    tabType,
    id,
    ...expandedDetails
  } = action;
  const panelViewOptions = new Set(['eventDetail', 'hostDetail', 'networkDetail', 'userDetail']);
  const expandedTabType = tabType !== null && tabType !== void 0 ? tabType : 'query';
  const newExpandDetails = {
    params: expandedDetails.params ? {
      ...expandedDetails.params
    } : {},
    panelView: expandedDetails.panelView
  };
  return {
    [expandedTabType]: panelViewOptions.has((_expandedDetails$pane = expandedDetails.panelView) !== null && _expandedDetails$pane !== void 0 ? _expandedDetails$pane : '') ? newExpandDetails : {}
  };
};
exports.updateTimelineDetailsPanel = updateTimelineDetailsPanel;
const setLoadingTableEvents = ({
  id,
  eventIds,
  isLoading,
  timelineById
}) => {
  const timeline = timelineById[id];
  const loadingEventIds = isLoading ? (0, _fp.union)(timeline.loadingEventIds, eventIds) : timeline.loadingEventIds.filter(currentEventId => !eventIds.includes(currentEventId));
  return {
    ...timelineById,
    [id]: {
      ...timeline,
      loadingEventIds
    }
  };
};
exports.setLoadingTableEvents = setLoadingTableEvents;
const removeTableColumn = ({
  id,
  columnId,
  timelineById
}) => {
  const timeline = timelineById[id];
  const columns = timeline.columns.filter(c => c.id !== columnId);
  return {
    ...timelineById,
    [id]: {
      ...timeline,
      columns
    }
  };
};

/**
 * Adds or updates a column. When updating a column, it will be moved to the
 * new index
 */
exports.removeTableColumn = removeTableColumn;
const upsertTableColumn = ({
  column,
  id,
  index,
  timelineById
}) => {
  const timeline = timelineById[id];
  const alreadyExistsAtIndex = timeline.columns.findIndex(c => c.id === column.id);
  if (alreadyExistsAtIndex !== -1) {
    // remove the existing entry and add the new one at the specified index
    const reordered = timeline.columns.filter(c => c.id !== column.id);
    reordered.splice(index, 0, column); // ⚠️ mutation

    return {
      ...timelineById,
      [id]: {
        ...timeline,
        columns: reordered
      }
    };
  }
  // add the new entry at the specified index
  const columns = [...timeline.columns];
  columns.splice(index, 0, column); // ⚠️ mutation

  return {
    ...timelineById,
    [id]: {
      ...timeline,
      columns
    }
  };
};
exports.upsertTableColumn = upsertTableColumn;
const updateTableColumns = ({
  id,
  columns,
  timelineById
}) => {
  const timeline = timelineById[id];
  return {
    ...timelineById,
    [id]: {
      ...timeline,
      columns
    }
  };
};
exports.updateTableColumns = updateTableColumns;
const updateTableSort = ({
  id,
  sort,
  timelineById
}) => {
  const timeline = timelineById[id];
  return {
    ...timelineById,
    [id]: {
      ...timeline,
      sort
    }
  };
};
exports.updateTableSort = updateTableSort;
const setSelectedTableEvents = ({
  id,
  eventIds,
  isSelectAllChecked = false,
  isSelected,
  timelineById
}) => {
  const timeline = timelineById[id];
  const selectedEventIds = isSelected ? {
    ...timeline.selectedEventIds,
    ...eventIds
  } : (0, _fp.omit)(Object.keys(eventIds), timeline.selectedEventIds);
  return {
    ...timelineById,
    [id]: {
      ...timeline,
      selectedEventIds,
      isSelectAllChecked
    }
  };
};
exports.setSelectedTableEvents = setSelectedTableEvents;
const setDeletedTableEvents = ({
  id,
  eventIds,
  isDeleted,
  timelineById
}) => {
  const timeline = timelineById[id];
  const deletedEventIds = isDeleted ? (0, _fp.union)(timeline.deletedEventIds, eventIds) : timeline.deletedEventIds.filter(currentEventId => !eventIds.includes(currentEventId));
  const selectedEventIds = Object.fromEntries(Object.entries(timeline.selectedEventIds).filter(([selectedEventId]) => !deletedEventIds.includes(selectedEventId)));
  const isSelectAllChecked = Object.keys(selectedEventIds).length > 0 ? timeline.isSelectAllChecked : false;
  return {
    ...timelineById,
    [id]: {
      ...timeline,
      deletedEventIds,
      selectedEventIds,
      isSelectAllChecked
    }
  };
};
exports.setDeletedTableEvents = setDeletedTableEvents;
const setInitializeTimelineSettings = ({
  id,
  timelineById,
  timelineSettingsProps
}) => {
  var _timelineSettingsProp;
  const timeline = timelineById[id];
  return !(timeline !== null && timeline !== void 0 && timeline.initialized) ? {
    ...timelineById,
    [id]: {
      ..._defaults.timelineDefaults,
      ...(0, _defaults.getTimelineManageDefaults)(id),
      ...timeline,
      ...timelineSettingsProps,
      ...(!timeline || (0, _fp.isEmpty)(timeline.columns) && !(0, _fp.isEmpty)(timelineSettingsProps.defaultColumns) ? {
        columns: timelineSettingsProps.defaultColumns
      } : {}),
      sort: (_timelineSettingsProp = timelineSettingsProps.sort) !== null && _timelineSettingsProp !== void 0 ? _timelineSettingsProp : _defaults.timelineDefaults.sort,
      loadingEventIds: _defaults.timelineDefaults.loadingEventIds,
      initialized: true
    }
  } : timelineById;
};
exports.setInitializeTimelineSettings = setInitializeTimelineSettings;
const applyDeltaToTableColumnWidth = ({
  id,
  columnId,
  delta,
  timelineById
}) => {
  var _timeline$columns$col2;
  const timeline = timelineById[id];
  const columnIndex = timeline.columns.findIndex(c => c.id === columnId);
  if (columnIndex === -1) {
    // the column was not found
    return {
      ...timelineById,
      [id]: {
        ...timeline
      }
    };
  }
  const requestedWidth = ((_timeline$columns$col2 = timeline.columns[columnIndex].initialWidth) !== null && _timeline$columns$col2 !== void 0 ? _timeline$columns$col2 : _constants.DEFAULT_COLUMN_MIN_WIDTH) + delta; // raw change in width
  const initialWidth = Math.max(_constants.RESIZED_COLUMN_MIN_WITH, requestedWidth); // if the requested width is smaller than the min, use the min

  const columnWithNewWidth = {
    ...timeline.columns[columnIndex],
    initialWidth
  };
  const columns = [...timeline.columns.slice(0, columnIndex), columnWithNewWidth, ...timeline.columns.slice(columnIndex + 1)];
  return {
    ...timelineById,
    [id]: {
      ...timeline,
      columns
    }
  };
};
exports.applyDeltaToTableColumnWidth = applyDeltaToTableColumnWidth;