"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.cameraInitialState = cameraInitialState;
exports.cameraReducer = void 0;
var _scaling_constants = require("./scaling_constants");
var _methods = require("./methods");
var vector2 = _interopRequireWildcard(require("../../models/vector2"));
var selectors = _interopRequireWildcard(require("./selectors"));
var _math = require("../../lib/math");
var _scale_to_zoom = require("./scale_to_zoom");
function _getRequireWildcardCache(nodeInterop) { if (typeof WeakMap !== "function") return null; var cacheBabelInterop = new WeakMap(); var cacheNodeInterop = new WeakMap(); return (_getRequireWildcardCache = function (nodeInterop) { return nodeInterop ? cacheNodeInterop : cacheBabelInterop; })(nodeInterop); }
function _interopRequireWildcard(obj, nodeInterop) { if (!nodeInterop && obj && obj.__esModule) { return obj; } if (obj === null || typeof obj !== "object" && typeof obj !== "function") { return { default: obj }; } var cache = _getRequireWildcardCache(nodeInterop); if (cache && cache.has(obj)) { return cache.get(obj); } var newObj = {}; var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var key in obj) { if (key !== "default" && Object.prototype.hasOwnProperty.call(obj, key)) { var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null; if (desc && (desc.get || desc.set)) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } newObj.default = obj; if (cache) { cache.set(obj, newObj); } return newObj; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Used in tests.
 */
function cameraInitialState() {
  const state = {
    scalingFactor: (0, _scale_to_zoom.scaleToZoom)(1),
    // Defaulted to 1 to 1 scale
    rasterSize: [0, 0],
    translationNotCountingCurrentPanning: [0, 0],
    latestFocusedWorldCoordinates: null,
    animation: undefined,
    panning: undefined
  };
  return state;
}
const cameraReducer = (state = cameraInitialState(), action) => {
  if (action.type === 'userSetZoomLevel') {
    /**
     * Handle the scale being explicitly set, for example by a 'reset zoom' feature, or by a range slider with exact scale values
     */

    const nextState = {
      ...state,
      scalingFactor: (0, _math.clamp)(action.payload, 0, 1)
    };
    return nextState;
  } else if (action.type === 'userClickedZoomIn') {
    return {
      ...state,
      scalingFactor: (0, _math.clamp)(state.scalingFactor + 0.1, 0, 1)
    };
  } else if (action.type === 'userClickedZoomOut') {
    return {
      ...state,
      scalingFactor: (0, _math.clamp)(state.scalingFactor - 0.1, 0, 1)
    };
  } else if (action.type === 'userZoomed') {
    const stateWithNewScaling = {
      ...state,
      scalingFactor: (0, _math.clamp)(state.scalingFactor + action.payload.zoomChange, 0, 1)
    };

    /**
     * Zooming fundamentally just changes the scale, but that would always zoom in (or out) around the center of the map. The user might be interested in
     * something else, like a node. If the user has moved their pointer on to the map, we can keep the pointer over the same point in the map by adjusting the
     * panning when we zoom.
     *
     * You can see this in action by moving your pointer over a node that isn't directly in the center of the map and then changing the zoom level. Do it by
     * using CTRL and the mousewheel, or by pinching the trackpad on a Mac. The node will stay under your mouse cursor and other things in the map will get
     * nearer or further from the mouse cursor. This lets you keep your context when changing zoom levels.
     */
    if (state.latestFocusedWorldCoordinates !== null && !selectors.isAnimating(state)(action.payload.time)) {
      const rasterOfLastFocusedWorldCoordinates = vector2.applyMatrix3(state.latestFocusedWorldCoordinates, selectors.projectionMatrix(state)(action.payload.time));
      const newWorldCoordinatesAtLastFocusedPosition = vector2.applyMatrix3(rasterOfLastFocusedWorldCoordinates, selectors.inverseProjectionMatrix(stateWithNewScaling)(action.payload.time));

      /**
       * The change in world position incurred by changing scale.
       */
      const delta = vector2.subtract(newWorldCoordinatesAtLastFocusedPosition, state.latestFocusedWorldCoordinates);

      /**
       * Adjust for the change in position due to scale.
       */
      const translationNotCountingCurrentPanning = vector2.subtract(stateWithNewScaling.translationNotCountingCurrentPanning, delta);
      const nextState = {
        ...stateWithNewScaling,
        translationNotCountingCurrentPanning
      };
      return nextState;
    } else {
      return stateWithNewScaling;
    }
  } else if (action.type === 'userSetPositionOfCamera') {
    /**
     * Handle the case where the position of the camera is explicitly set, for example by a 'back to center' feature.
     */
    const nextState = {
      ...state,
      animation: undefined,
      translationNotCountingCurrentPanning: action.payload
    };
    return nextState;
  } else if (action.type === 'userStartedPanning') {
    if (selectors.isAnimating(state)(action.payload.time)) {
      return state;
    }
    /**
     * When the user begins panning with a mousedown event we mark the starting position for later comparisons.
     */
    const nextState = {
      ...state,
      animation: undefined,
      panning: {
        origin: action.payload.screenCoordinates,
        currentOffset: action.payload.screenCoordinates
      }
    };
    return nextState;
  } else if (action.type === 'userStoppedPanning') {
    /**
     * When the user stops panning (by letting up on the mouse) we calculate the new translation of the camera.
     */
    const nextState = {
      ...state,
      translationNotCountingCurrentPanning: selectors.translation(state)(action.payload.time),
      panning: undefined
    };
    return nextState;
  } else if (action.type === 'userNudgedCamera') {
    const {
      direction,
      time
    } = action.payload;
    /**
     * Nudge less when zoomed in.
     */
    const nudge = vector2.multiply(vector2.divide([_scaling_constants.unitsPerNudge, _scaling_constants.unitsPerNudge], selectors.scale(state)(time)), direction);
    return (0, _methods.animatePanning)(state, time, vector2.add(state.translationNotCountingCurrentPanning, nudge), _scaling_constants.nudgeAnimationDuration);
  } else if (action.type === 'userSetRasterSize') {
    /**
     * Handle resizes of the Resolver component. We need to know the size in order to convert between screen
     * and world coordinates.
     */
    const nextState = {
      ...state,
      rasterSize: action.payload
    };
    return nextState;
  } else if (action.type === 'userMovedPointer') {
    let stateWithUpdatedPanning = state;
    if (state.panning) {
      stateWithUpdatedPanning = {
        ...state,
        panning: {
          origin: state.panning.origin,
          currentOffset: action.payload.screenCoordinates
        }
      };
    }
    const nextState = {
      ...stateWithUpdatedPanning,
      /**
       * keep track of the last world coordinates the user moved over.
       * When the scale of the projection matrix changes, we adjust the camera's world transform in order
       * to keep the same point under the pointer.
       * In order to do this, we need to know the position of the mouse when changing the scale.
       */
      latestFocusedWorldCoordinates: vector2.applyMatrix3(action.payload.screenCoordinates, selectors.inverseProjectionMatrix(stateWithUpdatedPanning)(action.payload.time))
    };
    return nextState;
  } else {
    return state;
  }
};
exports.cameraReducer = cameraReducer;