"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.children = children;
exports.factory = factory;
exports.levelOrder = levelOrder;
exports.parent = parent;
exports.root = root;
exports.size = size;
exports.treeNode = treeNode;
var _process_event = require("../process_event");
var _tree_sequencers = require("../../lib/tree_sequencers");
var nodeModel = _interopRequireWildcard(require("../../../../common/endpoint/models/node"));
function _getRequireWildcardCache(nodeInterop) { if (typeof WeakMap !== "function") return null; var cacheBabelInterop = new WeakMap(); var cacheNodeInterop = new WeakMap(); return (_getRequireWildcardCache = function (nodeInterop) { return nodeInterop ? cacheNodeInterop : cacheBabelInterop; })(nodeInterop); }
function _interopRequireWildcard(obj, nodeInterop) { if (!nodeInterop && obj && obj.__esModule) { return obj; } if (obj === null || typeof obj !== "object" && typeof obj !== "function") { return { default: obj }; } var cache = _getRequireWildcardCache(nodeInterop); if (cache && cache.has(obj)) { return cache.get(obj); } var newObj = {}; var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var key in obj) { if (key !== "default" && Object.prototype.hasOwnProperty.call(obj, key)) { var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null; if (desc && (desc.get || desc.set)) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } newObj.default = obj; if (cache) { cache.set(obj, newObj); } return newObj; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function calculateGenerationsAndDescendantsFromOrigin(origin, descendants) {
  if (!origin) {
    return;
  }
  return (0, _tree_sequencers.calculateGenerationsAndDescendants)({
    node: origin,
    currentLevel: 0,
    totalDescendants: 0,
    children: parentNode => {
      var _descendants$get;
      return (_descendants$get = descendants.get(nodeModel.nodeID(parentNode))) !== null && _descendants$get !== void 0 ? _descendants$get : [];
    }
  });
}
function parentInternal(node, idToNode) {
  const uniqueParentId = nodeModel.parentId(node);
  if (uniqueParentId === undefined) {
    return undefined;
  } else {
    return idToNode.get(uniqueParentId);
  }
}

/**
 * Returns the number of ancestors nodes (including the origin) in the graph.
 */
function countAncestors(originID, idToNode) {
  if (!originID) {
    return;
  }

  // include the origin
  let total = 1;
  let current = idToNode.get(originID);
  while (current !== undefined && parentInternal(current, idToNode) !== undefined) {
    total++;
    current = parentInternal(current, idToNode);
  }
  return total;
}

/**
 * Create a new IndexedProcessTree from an array of ProcessEvents.
 * siblings will be ordered by timestamp
 */
function factory(
// Array of processes to index as a tree
nodes, originID) {
  const idToChildren = new Map();
  const idToValue = new Map();
  for (const node of nodes) {
    const nodeID = nodeModel.nodeID(node);
    if (nodeID !== undefined) {
      idToValue.set(nodeID, node);
      const uniqueParentId = nodeModel.parentId(node);
      let childrenWithTheSameParent = idToChildren.get(uniqueParentId);
      if (!childrenWithTheSameParent) {
        childrenWithTheSameParent = [];
        idToChildren.set(uniqueParentId, childrenWithTheSameParent);
      }
      childrenWithTheSameParent.push(node);
    }
  }

  // sort the children of each node
  for (const siblings of idToChildren.values()) {
    siblings.sort(_process_event.orderByTime);
  }
  let generations;
  let descendants;
  if (originID) {
    const originNode = idToValue.get(originID);
    const treeGenerationsAndDescendants = calculateGenerationsAndDescendantsFromOrigin(originNode, idToChildren);
    generations = treeGenerationsAndDescendants === null || treeGenerationsAndDescendants === void 0 ? void 0 : treeGenerationsAndDescendants.generations;
    descendants = treeGenerationsAndDescendants === null || treeGenerationsAndDescendants === void 0 ? void 0 : treeGenerationsAndDescendants.descendants;
  }
  const ancestors = countAncestors(originID, idToValue);
  return {
    idToChildren,
    idToNode: idToValue,
    originID,
    generations,
    descendants,
    ancestors
  };
}

/**
 * Returns an array with any children `ProcessEvent`s of the passed in `process`
 */
function children(tree, parentID) {
  const currentSiblings = tree.idToChildren.get(parentID);
  return currentSiblings === undefined ? [] : currentSiblings;
}

/**
 * Get the indexed process event for the ID
 */
function treeNode(tree, entityID) {
  var _tree$idToNode$get;
  return (_tree$idToNode$get = tree.idToNode.get(entityID)) !== null && _tree$idToNode$get !== void 0 ? _tree$idToNode$get : null;
}

/**
 * Returns the parent ProcessEvent, if any, for the passed in `childProcess`
 */
function parent(tree, childNode) {
  return parentInternal(childNode, tree.idToNode);
}

/**
 * Number of processes in the tree
 */
function size(tree) {
  return tree.idToNode.size;
}

/**
 * Return the root process
 */
function root(tree) {
  if (size(tree) === 0) {
    return null;
  }
  // any node will do
  let current = tree.idToNode.values().next().value;

  // iteratively swap current w/ its parent
  while (parent(tree, current) !== undefined) {
    // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
    current = parent(tree, current);
  }
  return current;
}

/**
 * Yield processes in level order
 */
function* levelOrder(tree) {
  const rootNode = root(tree);
  if (rootNode !== null) {
    yield* (0, _tree_sequencers.levelOrder)(rootNode, parentNode => children(tree, nodeModel.nodeID(parentNode)));
  }
}