"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ExceptionsListApiClient = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _securitysolutionListConstants = require("@kbn/securitysolution-list-constants");
var _constants = require("../../common/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * A generic class to be used for each artifact type.
 * It follow the Singleton pattern.
 * Please, use the getInstance method instead of creating a new instance when using this implementation.
 */
class ExceptionsListApiClient {
  constructor(http, listId, listDefinition, readTransform, writeTransform) {
    (0, _defineProperty2.default)(this, "ensureListExists", void 0);
    this.http = http;
    this.listId = listId;
    this.listDefinition = listDefinition;
    this.readTransform = readTransform;
    this.writeTransform = writeTransform;
    this.ensureListExists = this.createExceptionList();
  }

  /**
   * PrivateStatic method that creates the list.
   * This method is being used when initializing an instance only once.
   */
  async createExceptionList() {
    if (ExceptionsListApiClient.wasListCreated.has(this.listId)) {
      return ExceptionsListApiClient.wasListCreated.get(this.listId);
    }
    ExceptionsListApiClient.wasListCreated.set(this.listId, new Promise((resolve, reject) => {
      const asyncFunction = async () => {
        try {
          await this.http.post(_securitysolutionListConstants.INTERNAL_EXCEPTIONS_LIST_ENSURE_CREATED_URL, {
            body: JSON.stringify({
              ...this.listDefinition,
              list_id: this.listId
            })
          });
          resolve();
        } catch (err) {
          ExceptionsListApiClient.wasListCreated.delete(this.listId);
          reject(err);
        }
      };
      asyncFunction();
    }));
    return ExceptionsListApiClient.wasListCreated.get(this.listId);
  }

  /**
   * Private method that throws an error when some of the checks to ensure the instance
   * we are using is the right one fail
   */
  checkIfIsUsingTheRightInstance(listId) {
    if (listId !== this.listId) {
      throw new Error(`The list id you are using is not valid, expected [${this.listId}] list id but received [${listId}] list id`);
    }
  }
  isHttp(coreHttp) {
    return this.http === coreHttp;
  }
  getHttp() {
    return this.http;
  }

  /**
   * Static method to get a fresh or existing instance.
   * It will ensure we only check and create the list once.
   */
  static getInstance(http, listId, listDefinition, readTransform, writeTransform) {
    var _ExceptionsListApiCli;
    if (!ExceptionsListApiClient.instance.has(listId) || !((_ExceptionsListApiCli = ExceptionsListApiClient.instance.get(listId)) !== null && _ExceptionsListApiCli !== void 0 && _ExceptionsListApiCli.isHttp(http))) {
      ExceptionsListApiClient.instance.set(listId, new ExceptionsListApiClient(http, listId, listDefinition, readTransform, writeTransform));
    }
    const currentInstance = ExceptionsListApiClient.instance.get(listId);
    if (currentInstance) {
      return currentInstance;
    } else {
      return new ExceptionsListApiClient(http, listId, listDefinition, readTransform, writeTransform);
    }
  }

  /**
   * Static method to clean an exception item before sending it to update.
   */
  static cleanExceptionsBeforeUpdate(exception) {
    var _exceptionToUpdateCle;
    const exceptionToUpdateCleaned = {
      ...exception
    };
    // Clean unnecessary fields for update action
    ['created_at', 'created_by', 'list_id', 'tie_breaker_id', 'updated_at', 'updated_by', 'meta'].forEach(field => {
      delete exceptionToUpdateCleaned[field];
    });
    exceptionToUpdateCleaned.comments = (_exceptionToUpdateCle = exceptionToUpdateCleaned.comments) === null || _exceptionToUpdateCle === void 0 ? void 0 : _exceptionToUpdateCle.map(comment => ({
      comment: comment.comment,
      id: comment.id
    }));
    return exceptionToUpdateCleaned;
  }

  /**
   * Returns a list of items with pagination params.
   * It accepts the allowed filtering, sorting and pagination options as param.
   */
  async find({
    perPage = _constants.MANAGEMENT_DEFAULT_PAGE_SIZE,
    page = _constants.MANAGEMENT_DEFAULT_PAGE + 1,
    sortField,
    sortOrder,
    filter
  } = {}) {
    await this.ensureListExists;
    const result = await this.http.get(`${_securitysolutionListConstants.EXCEPTION_LIST_ITEM_URL}/_find`, {
      query: {
        page,
        per_page: perPage,
        sort_field: sortField,
        sort_order: sortOrder,
        list_id: [this.listId],
        namespace_type: ['agnostic'],
        filter
      }
    });
    if (this.readTransform) {
      result.data = result.data.map(this.readTransform);
    }
    return result;
  }

  /**
   * Returns an item for the given `itemId` or `id`. Exception List Items have both an `item_id`
   * and `id`, and at least one of these two is required to be provided.
   */
  async get(itemId, id) {
    if (!itemId && !id) {
      throw TypeError('either `itemId` or `id` argument must be set');
    }
    await this.ensureListExists;
    let result = await this.http.get(_securitysolutionListConstants.EXCEPTION_LIST_ITEM_URL, {
      query: {
        id,
        item_id: itemId,
        namespace_type: 'agnostic'
      }
    });
    if (this.readTransform) {
      result = this.readTransform(result);
    }
    return result;
  }

  /**
   * It creates an item and returns the created one.
   * It requires a CreateExceptionListItemSchema object.
   */
  async create(exception) {
    await this.ensureListExists;
    this.checkIfIsUsingTheRightInstance(exception.list_id);
    delete exception.meta;
    let transformedException = exception;
    if (this.writeTransform) {
      transformedException = this.writeTransform(exception);
    }
    return this.http.post(_securitysolutionListConstants.EXCEPTION_LIST_ITEM_URL, {
      body: JSON.stringify(transformedException)
    });
  }

  /**
   * It updates an existing item and returns the updated one.
   * It requires a UpdateExceptionListItemSchema object.
   */
  async update(exception) {
    await this.ensureListExists;
    let transformedException = exception;
    if (this.writeTransform) {
      transformedException = this.writeTransform(exception);
    }
    return this.http.put(_securitysolutionListConstants.EXCEPTION_LIST_ITEM_URL, {
      body: JSON.stringify(ExceptionsListApiClient.cleanExceptionsBeforeUpdate(transformedException))
    });
  }

  /**
   * It deletes an existing item by `itemId` or `id`. Exception List Items have both an `item_id`
   * and `id`, and at least one of these two is required to be provided.
   */
  async delete(itemId, id) {
    if (!itemId && !id) {
      throw TypeError('either `itemId` or `id` argument must be set');
    }
    await this.ensureListExists;
    return this.http.delete(_securitysolutionListConstants.EXCEPTION_LIST_ITEM_URL, {
      query: {
        id,
        item_id: itemId,
        namespace_type: 'agnostic'
      }
    });
  }

  /**
   * It returns a summary of the current list_id
   * It accepts a filter param to filter the summary results using KQL filtering.
   */
  async summary(filter) {
    await this.ensureListExists;
    return this.http.get(`${_securitysolutionListConstants.EXCEPTION_LIST_URL}/summary`, {
      query: {
        filter,
        list_id: this.listId,
        namespace_type: 'agnostic'
      }
    });
  }

  /**
   * Checks if the given list has any data in it
   */
  async hasData() {
    return (await this.find({
      perPage: 1,
      page: 1
    })).total > 0;
  }
}
exports.ExceptionsListApiClient = ExceptionsListApiClient;
(0, _defineProperty2.default)(ExceptionsListApiClient, "instance", new Map());
(0, _defineProperty2.default)(ExceptionsListApiClient, "wasListCreated", new Map());