"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getRbacControl = exports.getEndpointConsoleCommands = exports.HELP_GROUPS = void 0;
var _i18n = require("@kbn/i18n");
var _experimental_features_service = require("../../../../common/experimental_features_service");
var _get_file_action = require("../command_render_components/get_file_action");
var _isolate_action = require("../command_render_components/isolate_action");
var _release_action = require("../command_render_components/release_action");
var _kill_process_action = require("../command_render_components/kill_process_action");
var _suspend_process_action = require("../command_render_components/suspend_process_action");
var _status_action = require("../command_render_components/status_action");
var _get_processes_action = require("../command_render_components/get_processes_action");
var _execute_action = require("../command_render_components/execute_action");
var _translations = require("../../../../common/translations");
var _get_command_about_info = require("./get_command_about_info");
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const emptyArgumentValidator = argData => {
  var _argData$;
  if ((argData === null || argData === void 0 ? void 0 : argData.length) > 0 && typeof argData[0] === 'string' && ((_argData$ = argData[0]) === null || _argData$ === void 0 ? void 0 : _argData$.trim().length) > 0) {
    return true;
  } else {
    return _i18n.i18n.translate('xpack.securitySolution.endpointConsoleCommands.emptyArgumentMessage', {
      defaultMessage: 'Argument cannot be empty'
    });
  }
};
const pidValidator = argData => {
  const emptyResult = emptyArgumentValidator(argData);
  if (emptyResult !== true) {
    return emptyResult;
  } else if (Number.isSafeInteger(Number(argData)) && Number(argData) > 0) {
    return true;
  } else {
    return _i18n.i18n.translate('xpack.securitySolution.endpointConsoleCommands.invalidPidMessage', {
      defaultMessage: 'Argument must be a positive number representing the PID of a process'
    });
  }
};
const executeTimeoutValidator = argData => {
  if (String(argData).trim().length && (0, _utils.validateUnitOfTime)(String(argData).trim())) {
    return true;
  } else {
    return _i18n.i18n.translate('xpack.securitySolution.endpointConsoleCommands.invalidExecuteTimeout', {
      defaultMessage: 'Argument must be a string with a positive integer value followed by a unit of time (h for hours, m for minutes, s for seconds). Example: 37m.'
    });
  }
};
const commandToCapabilitiesPrivilegesMap = new Map([['isolate', {
  capability: 'isolation',
  privilege: privileges => privileges.canIsolateHost
}], ['release', {
  capability: 'isolation',
  privilege: privileges => privileges.canUnIsolateHost
}], ['kill-process', {
  capability: 'kill_process',
  privilege: privileges => privileges.canKillProcess
}], ['suspend-process', {
  capability: 'suspend_process',
  privilege: privileges => privileges.canSuspendProcess
}], ['processes', {
  capability: 'running_processes',
  privilege: privileges => privileges.canGetRunningProcesses
}], ['get-file', {
  capability: 'get_file',
  privilege: privileges => privileges.canWriteFileOperations
}], ['execute', {
  capability: 'execute',
  privilege: privileges => privileges.canWriteExecuteOperations
}]]);
const getRbacControl = ({
  commandName,
  privileges
}) => {
  var _commandToCapabilitie;
  return Boolean((_commandToCapabilitie = commandToCapabilitiesPrivilegesMap.get(commandName)) === null || _commandToCapabilitie === void 0 ? void 0 : _commandToCapabilitie.privilege(privileges));
};
exports.getRbacControl = getRbacControl;
const capabilitiesAndPrivilegesValidator = command => {
  var _commandToCapabilitie2;
  const privileges = command.commandDefinition.meta.privileges;
  const endpointCapabilities = command.commandDefinition.meta.capabilities;
  const commandName = command.commandDefinition.name;
  const responderCapability = (_commandToCapabilitie2 = commandToCapabilitiesPrivilegesMap.get(commandName)) === null || _commandToCapabilitie2 === void 0 ? void 0 : _commandToCapabilitie2.capability;
  let errorMessage = '';
  if (!responderCapability) {
    errorMessage = errorMessage.concat(_translations.UPGRADE_ENDPOINT_FOR_RESPONDER);
  }
  if (responderCapability) {
    if (!endpointCapabilities.includes(responderCapability)) {
      errorMessage = errorMessage.concat(_translations.UPGRADE_ENDPOINT_FOR_RESPONDER);
    }
  }
  if (getRbacControl({
    commandName,
    privileges
  }) !== true) {
    errorMessage = errorMessage.concat(_translations.INSUFFICIENT_PRIVILEGES_FOR_COMMAND);
  }
  if (errorMessage.length) {
    return errorMessage;
  }
  return true;
};
const HELP_GROUPS = Object.freeze({
  responseActions: {
    position: 0,
    label: _i18n.i18n.translate('xpack.securitySolution.endpointConsoleCommands.groups.responseActions', {
      defaultMessage: 'Response actions'
    })
  }
});
exports.HELP_GROUPS = HELP_GROUPS;
const ENTER_PID_OR_ENTITY_ID_INSTRUCTION = _i18n.i18n.translate('xpack.securitySolution.endpointResponseActionsConsoleCommands.enterPidOrEntityId', {
  defaultMessage: 'Enter a pid or an entity id to execute'
});
const ENTER_OR_ADD_COMMENT_ARG_INSTRUCTION = _i18n.i18n.translate('xpack.securitySolution.endpointResponseActionsConsoleCommands.enterOrAddOptionalComment', {
  defaultMessage: 'Hit enter to execute or add an optional comment'
});
const COMMENT_ARG_ABOUT = _i18n.i18n.translate('xpack.securitySolution.endpointConsoleCommands.suspendProcess.commandArgAbout', {
  defaultMessage: 'A comment to go along with the action'
});
const getEndpointConsoleCommands = ({
  endpointAgentId,
  endpointCapabilities,
  endpointPrivileges
}) => {
  const isGetFileEnabled = _experimental_features_service.ExperimentalFeaturesService.get().responseActionGetFileEnabled;
  const isExecuteEnabled = _experimental_features_service.ExperimentalFeaturesService.get().responseActionExecuteEnabled;
  const doesEndpointSupportCommand = commandName => {
    var _commandToCapabilitie3;
    const responderCapability = (_commandToCapabilitie3 = commandToCapabilitiesPrivilegesMap.get(commandName)) === null || _commandToCapabilitie3 === void 0 ? void 0 : _commandToCapabilitie3.capability;
    if (responderCapability) {
      return endpointCapabilities.includes(responderCapability);
    }
    return false;
  };
  const consoleCommands = [{
    name: 'isolate',
    about: (0, _get_command_about_info.getCommandAboutInfo)({
      aboutInfo: _i18n.i18n.translate('xpack.securitySolution.endpointConsoleCommands.isolate.about', {
        defaultMessage: 'Isolate the host'
      }),
      isSupported: doesEndpointSupportCommand('isolate')
    }),
    RenderComponent: _isolate_action.IsolateActionResult,
    meta: {
      endpointId: endpointAgentId,
      capabilities: endpointCapabilities,
      privileges: endpointPrivileges
    },
    exampleUsage: 'isolate --comment "isolate this host"',
    exampleInstruction: ENTER_OR_ADD_COMMENT_ARG_INSTRUCTION,
    validate: capabilitiesAndPrivilegesValidator,
    args: {
      comment: {
        required: false,
        allowMultiples: false,
        about: COMMENT_ARG_ABOUT
      }
    },
    helpGroupLabel: HELP_GROUPS.responseActions.label,
    helpGroupPosition: HELP_GROUPS.responseActions.position,
    helpCommandPosition: 0,
    helpDisabled: doesEndpointSupportCommand('isolate') === false,
    helpHidden: !getRbacControl({
      commandName: 'isolate',
      privileges: endpointPrivileges
    })
  }, {
    name: 'release',
    about: (0, _get_command_about_info.getCommandAboutInfo)({
      aboutInfo: _i18n.i18n.translate('xpack.securitySolution.endpointConsoleCommands.release.about', {
        defaultMessage: 'Release the host'
      }),
      isSupported: doesEndpointSupportCommand('release')
    }),
    RenderComponent: _release_action.ReleaseActionResult,
    meta: {
      endpointId: endpointAgentId,
      capabilities: endpointCapabilities,
      privileges: endpointPrivileges
    },
    exampleUsage: 'release --comment "release this host"',
    exampleInstruction: ENTER_OR_ADD_COMMENT_ARG_INSTRUCTION,
    validate: capabilitiesAndPrivilegesValidator,
    args: {
      comment: {
        required: false,
        allowMultiples: false,
        about: COMMENT_ARG_ABOUT
      }
    },
    helpGroupLabel: HELP_GROUPS.responseActions.label,
    helpGroupPosition: HELP_GROUPS.responseActions.position,
    helpCommandPosition: 1,
    helpDisabled: doesEndpointSupportCommand('release') === false,
    helpHidden: !getRbacControl({
      commandName: 'release',
      privileges: endpointPrivileges
    })
  }, {
    name: 'kill-process',
    about: (0, _get_command_about_info.getCommandAboutInfo)({
      aboutInfo: _i18n.i18n.translate('xpack.securitySolution.endpointConsoleCommands.killProcess.about', {
        defaultMessage: 'Kill/terminate a process'
      }),
      isSupported: doesEndpointSupportCommand('kill-process')
    }),
    RenderComponent: _kill_process_action.KillProcessActionResult,
    meta: {
      endpointId: endpointAgentId,
      capabilities: endpointCapabilities,
      privileges: endpointPrivileges
    },
    exampleUsage: 'kill-process --pid 123 --comment "kill this process"',
    exampleInstruction: ENTER_PID_OR_ENTITY_ID_INSTRUCTION,
    validate: capabilitiesAndPrivilegesValidator,
    mustHaveArgs: true,
    args: {
      comment: {
        required: false,
        allowMultiples: false,
        about: COMMENT_ARG_ABOUT
      },
      pid: {
        required: false,
        allowMultiples: false,
        exclusiveOr: true,
        about: _i18n.i18n.translate('xpack.securitySolution.endpointConsoleCommands.pid.arg.comment', {
          defaultMessage: 'A PID representing the process to kill'
        }),
        validate: pidValidator
      },
      entityId: {
        required: false,
        allowMultiples: false,
        exclusiveOr: true,
        about: _i18n.i18n.translate('xpack.securitySolution.endpointConsoleCommands.entityId.arg.comment', {
          defaultMessage: 'An entity id representing the process to kill'
        }),
        validate: emptyArgumentValidator
      }
    },
    helpGroupLabel: HELP_GROUPS.responseActions.label,
    helpGroupPosition: HELP_GROUPS.responseActions.position,
    helpCommandPosition: 4,
    helpDisabled: doesEndpointSupportCommand('kill-process') === false,
    helpHidden: !getRbacControl({
      commandName: 'kill-process',
      privileges: endpointPrivileges
    })
  }, {
    name: 'suspend-process',
    about: (0, _get_command_about_info.getCommandAboutInfo)({
      aboutInfo: _i18n.i18n.translate('xpack.securitySolution.endpointConsoleCommands.suspendProcess.about', {
        defaultMessage: 'Temporarily suspend a process'
      }),
      isSupported: doesEndpointSupportCommand('suspend-process')
    }),
    RenderComponent: _suspend_process_action.SuspendProcessActionResult,
    meta: {
      endpointId: endpointAgentId,
      capabilities: endpointCapabilities,
      privileges: endpointPrivileges
    },
    exampleUsage: 'suspend-process --pid 123 --comment "suspend this process"',
    exampleInstruction: ENTER_PID_OR_ENTITY_ID_INSTRUCTION,
    validate: capabilitiesAndPrivilegesValidator,
    mustHaveArgs: true,
    args: {
      comment: {
        required: false,
        allowMultiples: false,
        about: COMMENT_ARG_ABOUT
      },
      pid: {
        required: false,
        allowMultiples: false,
        exclusiveOr: true,
        about: _i18n.i18n.translate('xpack.securitySolution.endpointConsoleCommands.suspendProcess.pid.arg.comment', {
          defaultMessage: 'A PID representing the process to suspend'
        }),
        validate: pidValidator
      },
      entityId: {
        required: false,
        allowMultiples: false,
        exclusiveOr: true,
        about: _i18n.i18n.translate('xpack.securitySolution.endpointConsoleCommands.suspendProcess.entityId.arg.comment', {
          defaultMessage: 'An entity id representing the process to suspend'
        }),
        validate: emptyArgumentValidator
      }
    },
    helpGroupLabel: HELP_GROUPS.responseActions.label,
    helpGroupPosition: HELP_GROUPS.responseActions.position,
    helpCommandPosition: 5,
    helpDisabled: doesEndpointSupportCommand('suspend-process') === false,
    helpHidden: !getRbacControl({
      commandName: 'suspend-process',
      privileges: endpointPrivileges
    })
  }, {
    name: 'status',
    about: _i18n.i18n.translate('xpack.securitySolution.endpointConsoleCommands.status.about', {
      defaultMessage: 'Show host status information'
    }),
    RenderComponent: _status_action.EndpointStatusActionResult,
    meta: {
      endpointId: endpointAgentId
    },
    helpGroupLabel: HELP_GROUPS.responseActions.label,
    helpGroupPosition: HELP_GROUPS.responseActions.position,
    helpCommandPosition: 2
  }, {
    name: 'processes',
    about: (0, _get_command_about_info.getCommandAboutInfo)({
      aboutInfo: _i18n.i18n.translate('xpack.securitySolution.endpointConsoleCommands.processes.about', {
        defaultMessage: 'Show all running processes'
      }),
      isSupported: doesEndpointSupportCommand('processes')
    }),
    RenderComponent: _get_processes_action.GetProcessesActionResult,
    meta: {
      endpointId: endpointAgentId,
      capabilities: endpointCapabilities,
      privileges: endpointPrivileges
    },
    exampleUsage: 'processes --comment "get the processes"',
    exampleInstruction: ENTER_OR_ADD_COMMENT_ARG_INSTRUCTION,
    validate: capabilitiesAndPrivilegesValidator,
    args: {
      comment: {
        required: false,
        allowMultiples: false,
        about: COMMENT_ARG_ABOUT
      }
    },
    helpGroupLabel: HELP_GROUPS.responseActions.label,
    helpGroupPosition: HELP_GROUPS.responseActions.position,
    helpCommandPosition: 3,
    helpDisabled: doesEndpointSupportCommand('processes') === false,
    helpHidden: !getRbacControl({
      commandName: 'processes',
      privileges: endpointPrivileges
    })
  }];

  // `get-file` is currently behind feature flag
  if (isGetFileEnabled) {
    consoleCommands.push({
      name: 'get-file',
      about: (0, _get_command_about_info.getCommandAboutInfo)({
        aboutInfo: _i18n.i18n.translate('xpack.securitySolution.endpointConsoleCommands.getFile.about', {
          defaultMessage: 'Retrieve a file from the host'
        }),
        isSupported: doesEndpointSupportCommand('processes')
      }),
      RenderComponent: _get_file_action.GetFileActionResult,
      meta: {
        endpointId: endpointAgentId,
        capabilities: endpointCapabilities,
        privileges: endpointPrivileges
      },
      exampleUsage: 'get-file --path "/full/path/to/file.txt" --comment "Possible malware"',
      exampleInstruction: ENTER_OR_ADD_COMMENT_ARG_INSTRUCTION,
      validate: capabilitiesAndPrivilegesValidator,
      mustHaveArgs: true,
      args: {
        path: {
          required: true,
          allowMultiples: false,
          about: _i18n.i18n.translate('xpack.securitySolution.endpointConsoleCommands.getFile.pathArgAbout', {
            defaultMessage: 'The full file path to be retrieved'
          }),
          validate: argData => {
            return emptyArgumentValidator(argData);
          }
        },
        comment: {
          required: false,
          allowMultiples: false,
          about: COMMENT_ARG_ABOUT
        }
      },
      helpGroupLabel: HELP_GROUPS.responseActions.label,
      helpGroupPosition: HELP_GROUPS.responseActions.position,
      helpCommandPosition: 6,
      helpDisabled: !doesEndpointSupportCommand('get-file'),
      helpHidden: !getRbacControl({
        commandName: 'get-file',
        privileges: endpointPrivileges
      })
    });
  }

  // `execute` is currently behind feature flag
  // planned for 8.8
  if (isExecuteEnabled) {
    consoleCommands.push({
      name: 'execute',
      about: (0, _get_command_about_info.getCommandAboutInfo)({
        aboutInfo: _i18n.i18n.translate('xpack.securitySolution.endpointConsoleCommands.execute.about', {
          defaultMessage: 'Execute a command on the host'
        }),
        isSupported: doesEndpointSupportCommand('execute')
      }),
      RenderComponent: _execute_action.ExecuteActionResult,
      meta: {
        endpointId: endpointAgentId,
        capabilities: endpointCapabilities,
        privileges: endpointPrivileges
      },
      exampleUsage: 'execute --command "ls -al" --timeout 2s --comment "Get list of all files"',
      exampleInstruction: ENTER_OR_ADD_COMMENT_ARG_INSTRUCTION,
      validate: capabilitiesAndPrivilegesValidator,
      mustHaveArgs: true,
      args: {
        command: {
          required: true,
          allowMultiples: false,
          about: (0, _execute_action.getExecuteCommandArgAboutInfo)(),
          mustHaveValue: 'non-empty-string'
        },
        timeout: {
          required: false,
          allowMultiples: false,
          about: _i18n.i18n.translate('xpack.securitySolution.endpointConsoleCommands.execute.args.timeout.about', {
            defaultMessage: 'The timeout in units of time (h for hours, m for minutes, s for seconds) for the endpoint to wait for the script to complete. Example: 37m. If not given, it defaults to 4 hours.'
          }),
          mustHaveValue: 'non-empty-string',
          validate: executeTimeoutValidator
        },
        comment: {
          required: false,
          allowMultiples: false,
          about: COMMENT_ARG_ABOUT
        }
      },
      helpGroupLabel: HELP_GROUPS.responseActions.label,
      helpGroupPosition: HELP_GROUPS.responseActions.position,
      helpCommandPosition: 6,
      helpDisabled: !doesEndpointSupportCommand('execute'),
      helpHidden: !getRbacControl({
        commandName: 'execute',
        privileges: endpointPrivileges
      })
    });
  }
  return consoleCommands;
};
exports.getEndpointConsoleCommands = getEndpointConsoleCommands;